package com.atlassian.bitbucket.event.pull;

import com.atlassian.bitbucket.comment.Comment;
import com.atlassian.bitbucket.comment.CommentAction;
import com.atlassian.bitbucket.event.CancelableEvent;
import com.atlassian.bitbucket.i18n.KeyedMessage;
import com.atlassian.bitbucket.pull.PullRequest;
import com.atlassian.bitbucket.util.CancelState;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import static java.util.Objects.requireNonNull;

/**
 * Event that is raised before a comment in a pull request is edited. This event is synchronous, allowing
 * listeners to perform operations in the same database transaction where the comment is edited.
 * <p>
 * This event is also {@link CancelableEvent cancelable}. A listener may prevent the comment from being edited by
 * {@link #cancel(KeyedMessage) canceling} this event. Throwing an exception <i>will not</i> prevent the comment from
 * being edited; the exception will be logged and ignored.
 *
 * @since 6.6
 */
public class PullRequestCommentModificationRequestedEvent extends PullRequestCancelableCommentEvent {

    private final String previousText;

    public PullRequestCommentModificationRequestedEvent(@Nonnull Object source, @Nonnull PullRequest pullRequest,
                                                        @Nonnull Comment comment, @Nullable Comment parent,
                                                        @Nonnull String previousText,
                                                        @Nonnull CancelState cancelState) {
        super(source, pullRequest, comment, parent, CommentAction.EDITED, cancelState);

        this.previousText = requireNonNull(previousText, "previousText");
    }

    @Nonnull
    public String getPreviousText() {
        return previousText;
    }
}
