package com.atlassian.bitbucket.event.attachment;

import com.atlassian.bitbucket.attachment.Attachment;
import com.atlassian.bitbucket.event.CancelableEvent;
import com.atlassian.bitbucket.event.annotation.TransactionAware;
import com.atlassian.bitbucket.i18n.KeyedMessage;
import com.atlassian.bitbucket.repository.Repository;
import com.atlassian.bitbucket.util.CancelState;

import javax.annotation.Nonnull;

import static java.util.Objects.requireNonNull;

/**
 * Raised just before an {@link Attachment attachment} is {@link AttachmentSavedEvent added}. This event is
 * synchronous, allowing listeners to perform operations in the same database transaction where the attachment will be
 * deleted.
 * <p>
 * This event is {@link CancelableEvent cancelable}. A listener may prevent the attachment from being added by
 * {@link #cancel(KeyedMessage) canceling} this event. Throwing an exception <i>will not</i> prevent the attachment
 * from being added; the exception will be logged and ignored.
 *
 * @since 7.0
 */
@TransactionAware(TransactionAware.When.IMMEDIATE)
public class AttachmentSaveRequestedEvent extends AttachmentEvent implements CancelableEvent {

    private final CancelState cancelState;

    /**
     * Construct a new {@code AttachmentEvent}.
     *
     * @param source       the component raising the event
     * @param repository   the repository
     * @param attachment   the attachment
     */
    public AttachmentSaveRequestedEvent(@Nonnull Object source, @Nonnull Repository repository,
                                        @Nonnull Attachment attachment, @Nonnull CancelState cancelState) {
        super(source, repository, attachment);

        this.cancelState = requireNonNull(cancelState, "cancelState");
    }

    /**
     * Cancels attachment addition, providing a message explaining why.
     *
     * @param message a descriptive message explaining why the operation has been canceled
     */
    @Override
    public void cancel(@Nonnull KeyedMessage message) {
        cancelState.cancel(message);
    }

    /**
     * Retrieves a flag indicating whether attachment addition has already been canceled by another listener.
     *
     * @return {@code true} if another listener has already canceled attachment addition; otherwise {@code false}
     */
    @Override
    public boolean isCanceled() {
        return cancelState.isCanceled();
    }
}
