package com.atlassian.bitbucket.commit;

import com.atlassian.bitbucket.repository.Repository;
import com.atlassian.bitbucket.util.BuilderSupport;

import javax.annotation.Nonnull;
import java.util.Objects;

import static java.util.Objects.requireNonNull;

/**
 * @since 5.0
 */
public class CommitDiscussionRequest {

    private final String commitId;
    private final boolean create;
    private final int hashCode;
    private final Repository repository;

    private CommitDiscussionRequest(Builder builder) {
        commitId = builder.commitId;
        create = builder.create;
        repository = builder.repository;

        //This will always be used at least once, so it's best to calculate it and keep it. The
        //hash code for a given key is immutable since the keys are themselves immutable.
        hashCode = Objects.hash(repository.getId(), commitId, create);
    }

    @Override
    public boolean equals(Object o) {
        if (o == this) {
            return true;
        }
        if (o instanceof CommitDiscussionRequest) {
            CommitDiscussionRequest r = (CommitDiscussionRequest) o;

            return repository.equals(r.getRepository()) &&
                    commitId.equals(r.getCommitId()) &&
                    create == r.isCreate();
        }
        return false;
    }

    @Nonnull
    public String getCommitId() {
        return commitId;
    }

    @Nonnull
    public Repository getRepository() {
        return repository;
    }

    @Override
    public int hashCode() {
        return hashCode;
    }

    public boolean isCreate() {
        return create;
    }

    public static class Builder extends BuilderSupport {

        private final String commitId;
        private final Repository repository;

        private boolean create;

        public Builder(@Nonnull Repository repository, @Nonnull String commitId) {
            this.repository = requireNonNull(repository, "repository");
            this.commitId = checkNotBlank(commitId, "commitId");
        }

        @Nonnull
        public CommitDiscussionRequest build() {
            return new CommitDiscussionRequest(this);
        }

        @Nonnull
        public Builder create(boolean value) {
            create = value;

            return this;
        }
    }
}
