/*
 * Decompiled with CFR 0.152.
 */
package com.atlassian.bitbucket.util;

import com.atlassian.bitbucket.util.FilePermission;
import com.atlassian.bitbucket.util.MoreCollectors;
import com.atlassian.bitbucket.util.SetFilePermissionRequest;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSet;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.NoSuchFileException;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.attribute.AclEntry;
import java.nio.file.attribute.AclEntryPermission;
import java.nio.file.attribute.AclEntryType;
import java.nio.file.attribute.AclFileAttributeView;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.nio.file.attribute.FileTime;
import java.nio.file.attribute.PosixFilePermission;
import java.nio.file.attribute.UserPrincipal;
import java.nio.file.attribute.UserPrincipalLookupService;
import java.nio.file.attribute.UserPrincipalNotFoundException;
import java.time.Instant;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Stream;
import javax.annotation.Nonnull;
import org.apache.commons.lang3.SystemUtils;
import org.apache.commons.lang3.mutable.MutableLong;

public class MoreFiles {
    private static final Set<AclEntryPermission> EXECUTE_ACL_PERMISSIONS = ImmutableSet.of((Object)((Object)AclEntryPermission.EXECUTE));
    private static final Set<AclEntryPermission> READ_ACL_PERMISSIONS = ImmutableSet.of((Object)((Object)AclEntryPermission.READ_ACL), (Object)((Object)AclEntryPermission.READ_ATTRIBUTES), (Object)((Object)AclEntryPermission.READ_DATA), (Object)((Object)AclEntryPermission.READ_NAMED_ATTRS));
    private static final Set<AclEntryPermission> WRITE_ACL_PERMISSIONS = ImmutableSet.of((Object)((Object)AclEntryPermission.WRITE_ACL), (Object)((Object)AclEntryPermission.WRITE_ATTRIBUTES), (Object)((Object)AclEntryPermission.WRITE_DATA), (Object)((Object)AclEntryPermission.WRITE_NAMED_ATTRS), (Object)((Object)AclEntryPermission.WRITE_OWNER), (Object)((Object)AclEntryPermission.APPEND_DATA), (Object[])new AclEntryPermission[]{AclEntryPermission.DELETE, AclEntryPermission.DELETE_CHILD, AclEntryPermission.SYNCHRONIZE});

    private MoreFiles() {
        throw new UnsupportedOperationException(this.getClass().getName() + " is a utility class and should not be instantiated");
    }

    public static void cleanDirectory(@Nonnull Path directory) throws IOException {
        Objects.requireNonNull(directory, "directory");
        try (Stream<Path> entries = Files.list(directory);){
            for (Path entry : entries::iterator) {
                BasicFileAttributes attributes = Files.readAttributes(entry, BasicFileAttributes.class, new LinkOption[0]);
                if (attributes.isDirectory()) {
                    MoreFiles.deleteRecursively(entry);
                    continue;
                }
                Files.delete(entry);
            }
        }
    }

    public static void deleteOnExit(@Nonnull Path path) {
        Objects.requireNonNull(path, "path").toFile().deleteOnExit();
    }

    public static boolean deleteQuietly(@Nonnull Path path) {
        try {
            MoreFiles.deleteRecursively(path);
        }
        catch (IOException ignored) {
            return false;
        }
        return true;
    }

    public static void deleteRecursively(@Nonnull Path path) throws IOException {
        Objects.requireNonNull(path, "path");
        Files.walkFileTree(path, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

            @Override
            public FileVisitResult postVisitDirectory(Path dir, IOException exc) throws IOException {
                return this.deleteIfExists(dir);
            }

            @Override
            public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                return this.deleteIfExists(file);
            }

            @Override
            public FileVisitResult visitFileFailed(Path file, IOException exc) throws IOException {
                if (exc instanceof FileNotFoundException || exc instanceof NoSuchFileException) {
                    return FileVisitResult.CONTINUE;
                }
                throw exc;
            }

            private FileVisitResult deleteIfExists(Path path) throws IOException {
                try {
                    Files.delete(path);
                }
                catch (FileNotFoundException | NoSuchFileException iOException) {
                    // empty catch block
                }
                return FileVisitResult.CONTINUE;
            }
        });
    }

    public static long getLastModified(@Nonnull Path path) {
        Objects.requireNonNull(path, "path");
        try {
            return Files.getLastModifiedTime(path, new LinkOption[0]).toMillis();
        }
        catch (IOException ignored) {
            return 0L;
        }
    }

    public static boolean isWithin(@Nonnull Path path, @Nonnull Path expectedParent) throws IOException {
        Path test;
        Objects.requireNonNull(path, "path");
        Objects.requireNonNull(expectedParent, "expectedParent");
        if (!Files.isDirectory(expectedParent, new LinkOption[0])) {
            throw new IllegalArgumentException("expectedParent is not a directory");
        }
        try {
            test = path.toRealPath(new LinkOption[0]);
        }
        catch (NoSuchFileException e) {
            test = Paths.get(path.toFile().getCanonicalPath(), new String[0]);
        }
        return test.startsWith(expectedParent.toRealPath(new LinkOption[0]));
    }

    @Nonnull
    public static Path mkdir(@Nonnull Path directory) {
        Objects.requireNonNull(directory, "directory");
        try {
            return Files.createDirectories(directory, new FileAttribute[0]);
        }
        catch (IOException e) {
            if (e instanceof FileAlreadyExistsException && Files.isDirectory(directory, new LinkOption[0])) {
                return directory;
            }
            throw new IllegalStateException("Could not create " + directory.toAbsolutePath(), e);
        }
    }

    @Nonnull
    public static Path mkdir(@Nonnull Path parent, @Nonnull String child) {
        Objects.requireNonNull(parent, "parent");
        if (Objects.requireNonNull(child, "child").trim().isEmpty()) {
            throw new IllegalArgumentException("A path for the created directory is required");
        }
        return MoreFiles.mkdir(parent.resolve(child));
    }

    public static void requireWithin(@Nonnull Path path, @Nonnull Path expectedParent) throws IOException {
        if (!MoreFiles.isWithin(path, expectedParent)) {
            throw new IllegalArgumentException(path + " is not contained within " + expectedParent);
        }
    }

    @Nonnull
    public static Path resolve(@Nonnull Path path, @Nonnull String first, String ... more) {
        return Objects.requireNonNull(path, "path").resolve(Paths.get(first, more));
    }

    public static void setPermissions(@Nonnull SetFilePermissionRequest request) throws IOException {
        Path path = Objects.requireNonNull(request, "request").getPath();
        if (!Files.isRegularFile(path, new LinkOption[0])) {
            throw new IllegalArgumentException("The provided file, " + path + ", does not exist or is not a file");
        }
        if (SystemUtils.IS_OS_WINDOWS) {
            Set<FilePermission> worldPermissions;
            AclFileAttributeView view = Files.getFileAttributeView(path, AclFileAttributeView.class, new LinkOption[0]);
            UserPrincipalLookupService principalService = path.getFileSystem().getUserPrincipalLookupService();
            ImmutableList.Builder entries = ImmutableList.builder();
            Set<FilePermission> ownerPermissions = request.getOwnerPermissions();
            if (!ownerPermissions.isEmpty()) {
                entries.add((Object)MoreFiles.createAclEntry(Files.getOwner(path, new LinkOption[0]), ownerPermissions));
                MoreFiles.createAclEntry(principalService, "administrators", ownerPermissions).ifPresent(arg_0 -> ((ImmutableList.Builder)entries).add(arg_0));
            }
            if (!(worldPermissions = request.getWorldPermissions()).isEmpty()) {
                MoreFiles.createAclEntry(principalService, "everyone", request.getWorldPermissions()).ifPresent(arg_0 -> ((ImmutableList.Builder)entries).add(arg_0));
            }
            view.setAcl((List<AclEntry>)entries.build());
        } else {
            try {
                Files.setPosixFilePermissions(path, MoreFiles.toPosixFilePermissions(request));
            }
            catch (UnsupportedOperationException e) {
                throw new IOException("Failed to update file permissions for " + path.toAbsolutePath(), e);
            }
        }
    }

    public static long size(@Nonnull Path path) {
        Objects.requireNonNull(path, "path");
        try {
            return Files.size(path);
        }
        catch (IOException e) {
            return 0L;
        }
    }

    public static long totalSize(@Nonnull Path path) {
        Objects.requireNonNull(path, "path");
        try {
            final MutableLong size = new MutableLong();
            Files.walkFileTree(path, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

                @Override
                public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) {
                    size.add(attrs.size());
                    return FileVisitResult.CONTINUE;
                }
            });
            return size.longValue();
        }
        catch (IOException ignored) {
            return 0L;
        }
    }

    public static void touch(@Nonnull Path path) throws IOException {
        try {
            Files.setLastModifiedTime(path, FileTime.from(Instant.now()));
        }
        catch (FileNotFoundException | NoSuchFileException e) {
            Files.createFile(path, new FileAttribute[0]);
        }
    }

    @Nonnull
    public static String toString(@Nonnull Path path) throws IOException {
        return MoreFiles.toString(path, StandardCharsets.UTF_8);
    }

    @Nonnull
    public static String toString(@Nonnull Path path, @Nonnull Charset charset) throws IOException {
        Objects.requireNonNull(path, "path");
        Objects.requireNonNull(charset, "charset");
        return new String(Files.readAllBytes(path), charset);
    }

    @Nonnull
    public static Path write(@Nonnull Path path, @Nonnull String value, OpenOption ... options) throws IOException {
        return MoreFiles.write(path, value, StandardCharsets.UTF_8, options);
    }

    @Nonnull
    public static Path write(@Nonnull Path path, @Nonnull String value, @Nonnull Charset charset, OpenOption ... options) throws IOException {
        Objects.requireNonNull(path, "path");
        Objects.requireNonNull(value, "value");
        Objects.requireNonNull(charset, "charset");
        return Files.write(path, value.getBytes(charset), options);
    }

    private static AclEntry createAclEntry(UserPrincipal principal, Set<FilePermission> permissions) {
        return AclEntry.newBuilder().setPermissions(permissions.stream().flatMap(permission -> {
            switch (permission) {
                case EXECUTE: {
                    return EXECUTE_ACL_PERMISSIONS.stream();
                }
                case READ: {
                    return READ_ACL_PERMISSIONS.stream();
                }
            }
            return WRITE_ACL_PERMISSIONS.stream();
        }).collect(MoreCollectors.toImmutableSet())).setPrincipal(principal).setType(AclEntryType.ALLOW).build();
    }

    private static Optional<AclEntry> createAclEntry(UserPrincipalLookupService principalService, String principalName, Set<FilePermission> permissions) throws IOException {
        try {
            return Optional.of(MoreFiles.createAclEntry(principalService.lookupPrincipalByGroupName(principalName), permissions));
        }
        catch (UserPrincipalNotFoundException e) {
            return Optional.empty();
        }
    }

    private static Set<PosixFilePermission> toPosixFilePermissions(SetFilePermissionRequest request) {
        ImmutableSet.Builder posixPermissions = ImmutableSet.builder();
        request.getOwnerPermissions().stream().map(permission -> {
            switch (permission) {
                case EXECUTE: {
                    return PosixFilePermission.OWNER_EXECUTE;
                }
                case READ: {
                    return PosixFilePermission.OWNER_READ;
                }
            }
            return PosixFilePermission.OWNER_WRITE;
        }).forEach(arg_0 -> ((ImmutableSet.Builder)posixPermissions).add(arg_0));
        request.getGroupPermissions().stream().map(permission -> {
            switch (permission) {
                case EXECUTE: {
                    return PosixFilePermission.GROUP_EXECUTE;
                }
                case READ: {
                    return PosixFilePermission.GROUP_READ;
                }
            }
            return PosixFilePermission.GROUP_WRITE;
        }).forEach(arg_0 -> ((ImmutableSet.Builder)posixPermissions).add(arg_0));
        request.getWorldPermissions().stream().map(permission -> {
            switch (permission) {
                case EXECUTE: {
                    return PosixFilePermission.OTHERS_EXECUTE;
                }
                case READ: {
                    return PosixFilePermission.OTHERS_READ;
                }
            }
            return PosixFilePermission.OTHERS_WRITE;
        }).forEach(arg_0 -> ((ImmutableSet.Builder)posixPermissions).add(arg_0));
        return posixPermissions.build();
    }
}

