package com.atlassian.bitbucket.util;

import javax.annotation.Nonnull;

import static com.google.common.base.Preconditions.checkArgument;

/**
 * Default implementation of {@link PageRequest}.
 */
public class PageRequestImpl implements PageRequest {

    private final int start;
    private final int limit;

    /**
     * Constructs a new {@code PageRequestImpl} for retrieving a page with {@code limit} entries from the specified
     * {@code start} index.
     *
     * @param start  the <i>zero-based</i> index of the first entry to include on the page
     * @param limit  the maximum number of entries to include on the page
     */
    public PageRequestImpl(int start, int limit) {
        this.start = start < 0 ? 0 : start;
        this.limit = limit <= 0 ? 1 : limit;
        checkArgument(this.limit <= MAX_PAGE_LIMIT, "Limit must be less than %s", MAX_PAGE_LIMIT);
    }

    @Override
    public int getStart() {
        return start;
    }

    @Override
    public int getLimit() {
        return limit;
    }

    @Override
    public int hashCode() {
        final int prime = 31;

        int result = prime + limit;
        result = prime * result + start;

        return result;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (getClass() != obj.getClass()) {
            return false;
        }
        PageRequestImpl other = (PageRequestImpl) obj;
        if (limit != other.limit) {
            return false;
        }
        if (start != other.start) {
            return false;
        }
        return true;
    }

    @Override
    public String toString() {
        return "PageRequestImpl{" +
                "start=" + start +
                ", limit=" + limit +
                '}';
    }

    @Nonnull
    @Override
    public PageRequest buildRestrictedPageRequest(int maxLimit) {
        return new PageRequestImpl(start, Math.min(limit, maxLimit));
    }
}
