package com.atlassian.bitbucket.server;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.Optional;

import static java.util.Optional.empty;
import static java.util.Optional.of;

/**
 * Identifies what mode the application is in. Currently there are only two modes: the {@link #DEFAULT default} mode
 * which causes the product to behave like a general-purpose code hosting platform and the {@link #MIRROR mirror} mode
 * which causes the product to behave like a code mirror for projects and repositories hosted in another server.
 *
 * @since 4.1
 */
public enum ApplicationMode {

    /**
     * Signifies the application should behave in the default mode - a general-purpose code hosting platform
     */
    DEFAULT("default"),

    /**
     * Signifies the application should behave like a code mirror for projects and repositories hosted in another server
     */
    MIRROR("mirror");

    private final String id;

    ApplicationMode(String id) {
        this.id = id;
    }

    @Nonnull
    public String getId() {
        return id;
    }

    /**
     * Coerces a string to an {@link ApplicationMode} or returns the supplied default value if this is not possible
     *
     * @param value the ID of the application mode
     * @return the corresponding {@link ApplicationMode} value
     */
    public static Optional<ApplicationMode> fromId(@Nullable String value) {
        if (value == null) {
            return empty();
        }

        for (ApplicationMode mode : values()) {
            if (mode.id.equalsIgnoreCase(value)) {
                return of(mode);
            }
        }
        return empty();
    }
}
