package com.atlassian.bitbucket.scm.pull;

import com.atlassian.bitbucket.commit.Commit;
import com.atlassian.bitbucket.commit.CommitCallback;
import com.atlassian.bitbucket.content.ChangeCallback;
import com.atlassian.bitbucket.content.DiffContentCallback;
import com.atlassian.bitbucket.pull.*;
import com.atlassian.bitbucket.repository.Branch;
import com.atlassian.bitbucket.scm.Command;

import javax.annotation.Nonnull;

/**
 * Creates {@link Command commands} which provide pull request functionality. Each method accepts a set of
 * {@code CommandParameters} which are used to control the command's behaviour and output.
 * <p>
 * <b>Plugin developers</b>: <i>This is probably not the interface you want to use.</i> {@link PullRequestService}
 * automatically chooses the correct SCM based on the repository, and provides a more stable, compatible API. The
 * documentation for each method on this interface includes a link to the API service method or methods which expose it.

 */
public interface ScmPullRequestCommandFactory {

    /**
     * @param parameters parameters describing how many changes to stream, and potentially for which paths
     * @param callback   a callback to receive changes
     * @return a command which, when executed, will stream the changes the pull request wants to apply
     * @see PullRequestService#streamChanges(PullRequestChangesRequest, ChangeCallback)
     */
    @Nonnull
    Command<Void> changes(@Nonnull PullRequestChangeCommandParameters parameters,
                          @Nonnull ChangeCallback callback);

    /**
     * @param callback a callback to receive commits
     * @return a command which, when executed, will stream the commits that are reachable from the
     *         {@link PullRequest#getFromRef() from ref} and not
     *         reachable from the {@link PullRequest#getToRef() to ref}
     * @see PullRequestService#streamCommits(int, long, CommitCallback)
     */
    @Nonnull
    Command<Void> commits(@Nonnull CommitCallback callback);

    /**
     * @param parameters parameters describing the paths to diff and what should be included in the diffs generated
     * @param callback   a callback to receive the diff
     * @return a command which, when executed, will stream the diff for the specified paths
     * @see PullRequestService#streamDiff(PullRequestDiffRequest, DiffContentCallback)
     */
    @Nonnull
    Command<Void> diff(@Nonnull PullRequestDiffCommandParameters parameters, @Nonnull DiffContentCallback callback);

    /**
     * Retrieves the effective diff for the pull request. The "effective diff" is comprised of the two commits that
     * should be compared to get a view of what changes the pull request includes. When new commits are added to
     * <i>either</i> of the branches involved in a pull request, the pull request's effective diff may change.
     * <p>
     * <b>Note</b>: This method is not exposed via the service API. It is only available directly on the SCM API.
     * The effective diff is something plugin developers generally do not need to be explicitly aware of. Instead,
     * the {@link #commits(CommitCallback) commits}, {@link #changes} and {@link #diff} should be retrieved using
     * their commands.
     *
     * @return a command which, when executed, will retrieve the {@link Commit#getId
     *         commit IDs} which describe the pull request's current effective diff
     */
    @Nonnull
    Command<PullRequestEffectiveDiff> effectiveDiff();

    /**
     * @param parameters parameters describing additional context which an SCM <i>may</i> consider when merging the
     *                   pull request, potentially altering the way final shape of the merge
     * @return a command which, when executed, will merge the pull request, applying its changes to the target branch
     * @see PullRequestService#merge(PullRequestMergeRequest)
     */
    @Nonnull
    Command<Branch> merge(@Nonnull PullRequestMergeCommandParameters parameters);

    /**
     * Attempt to merge the pull request to determine what the expected result of actually merging it would be.
     *
     * @param pullRequest the pull request to test for merge-ability
     * @return the result of trying to merge the pull request
     * @since 4.10
     */
    @Nonnull
    Command<PullRequestMergeResult> tryMerge(@Nonnull PullRequest pullRequest);
}
