package com.atlassian.bitbucket.scm;

import com.atlassian.bitbucket.commit.*;
import com.atlassian.bitbucket.commit.graph.TraversalCallback;
import com.atlassian.bitbucket.commit.graph.TraversalRequest;
import com.atlassian.bitbucket.content.*;
import com.atlassian.bitbucket.io.TypeAwareOutputSupplier;
import com.atlassian.bitbucket.repository.*;
import com.atlassian.bitbucket.util.Page;
import com.atlassian.bitbucket.util.PageRequest;

import javax.annotation.Nonnull;
import java.util.Map;

/**
 * Creates {@link Command commands} which provide basic SCM functionality such as creating repositories, retrieving
 * commits and viewing diffs. Each method accepts a set of {@code CommandParameters} which are used to control the
 * command's behaviour and output.
 * <p>
 * In general, commands fall into two categories:
 * <ul>
 *     <li>Paged: Accept a {@link PageRequest} describing the page of output that should be returned and return
 *     an object graph when the command is executed</li>
 *     <li>Streamed: Accept a callback to receive output and return {@code Void} (usually {@code null}) when the
 *     command is executed</li>
 * </ul>
 * Streaming commands are generally expected to handle their own paging, though there are some exceptions.
 * <p>
 * Note: The {@link Repository repository} against which created commands will operate was specified when the factory
 * was retrieved from the {@link ScmService#getCommandFactory(Repository) ScmService}, and cannot be changed.
 * <p>
 * <b>Plugin developers</b>: <i>This is probably not the interface you want to use.</i> The API provides services
 * whose functionality is backed by the SCM. Using those services automatically chooses the correct SCM based on
 * the repository, and provides a more stable, compatible API. In general, the functionality of this command factory
 * is exposed by:
 * <ul>
 *     <li>{@link CommitService CommitService}</li>
 *     <li>{@link ContentService ContentService}</li>
 *     <li>{@link RefService RefService}</li>
 * </ul>
 * The documentation for each method on this interface includes a link to the API service method or methods which
 * expose it.
 */
public interface ScmCommandFactory {

    /**
     * Retrieves blame (also sometimes referred to as annotations) for a file at a given revision.
     * <p>
     * The {@link PageRequest} provided determines the lines in the file for which blame is calculated,
     * and only lines within the scope of the page request will be returned. However, because adjacent lines may be
     * {@link Blame#getSpannedLines() spanned} by a single {@link Blame#getAuthor() author}, the number
     * of distinct {@link Blame} objects returned will frequently be less than the {@link PageRequest#getLimit() limit}.
     *
     * @param parameters  parameters describing the file and revision used to calculate blame
     * @param pageRequest describes the set of lines in the file to calculate blame for
     * @return a command which, when executed, will return the calculated {@link Blame} for the specified file and
     *         revision for the requested set of lines
     * @see BlameCommandParameters
     * @see ContentService#getBlame(Repository, String, String, PageRequest)
     * @see ContentService#streamFile(Repository, String, String, PageRequest, boolean, FileContentCallback)
     *
     * @since 5.0
     */
    @Nonnull
    Command<Page<Blame>> blame(@Nonnull BlameCommandParameters parameters, @Nonnull PageRequest pageRequest);

    /**
     * Retrieves a {@link Command} which, when executed, will stream {@link Branch branches} ordered either
     * alphabetically or by modification and optionally filtered.
     * <p>
     * When no explicit order is requested, the default ordering depends on whether a filter was provided:
     * <ul>
     *     <li>If filtering, branches default to {@link RefOrder#ALPHABETICAL alphabetical} order</li>
     *     <li>Otherwise, branches default to {@link RefOrder#MODIFICATION modification} order</li>
     * </ul>
     * Modification order retrieves branches based on the date of their most recent commit.
     *
     * @param parameters  parameters describing desired order and filter text
     * @param callback    the callback to receive streamed branches
     * @return a command which, when executed, will stream branches
     * @see BranchesCommandParameters
     * @see RefService#getBranches(RepositoryBranchesRequest, PageRequest)
     * @since 4.6
     */
    @Nonnull
    Command<Void> branches(@Nonnull BranchesCommandParameters parameters, @Nonnull BranchCallback callback);

    /**
     * Retrieves a {@link Page page} of {@link Branch branches} ordered either alphabetically or by modification and
     * optionally filtered.
     * <p>
     * When no explicit order is requested, the default ordering depends on whether a filter was provided:
     * <ul>
     *     <li>If filtering, branches default to {@link RefOrder#ALPHABETICAL alphabetical} order</li>
     *     <li>Otherwise, branches default to {@link RefOrder#MODIFICATION modification} order</li>
     * </ul>
     * Modification order retrieves branches based on the date of their most recent commit.
     *
     * @param parameters  parameters describing desired order and filter text
     * @param pageRequest describes the set of branches to return
     * @return a command which, when executed, will return the requested page of branches
     * @see BranchesCommandParameters
     * @see RefService#getBranches(RepositoryBranchesRequest, PageRequest)
     */
    @Nonnull
    Command<Page<Branch>> branches(@Nonnull BranchesCommandParameters parameters, @Nonnull PageRequest pageRequest);

    /**
     * Retrieves a {@link Command} which, when executed, will stream {@link Change changes} describing modifications
     * made between two revisions, optionally filtered by path.
     * <p>
     * If no explicit {@link ChangesCommandParameters#getSinceId() "since"} revision is specified, it defaults to the
     * parent of the {@link ChangesCommandParameters#getUntilId() "until"} revision. If the "until" revision specifies
     * a merge commit (a commit with multiple parents), <i>no changes will be returned</i>.
     * <p>
     * Note: Some {@link Change changes}, such as {@link ChangeType#COPY copies} and {@link ChangeType#MOVE moves},
     * affect more than one path. When filtering by path, if both paths are not available the change's type may not
     * be correctly detected by some SCMs.
     *
     * @param parameters parameters describing the revisions to compare and paths to filter by
     * @param callback   the callback to receive streamed commits
     * @return a command which, when executed, will stream changes
     * @see ChangesCommandParameters
     * @see CommitService#streamChanges(ChangesRequest, ChangeCallback)
     */
    @Nonnull
    Command<Void> changes(@Nonnull ChangesCommandParameters parameters, @Nonnull ChangeCallback callback);

    /**
     * Retrieves a {@link Page page} of {@link Change changes} describing the modifications made between two
     * revisions, optionally filtered by path.
     * <p>
     * If no explicit {@link ChangesCommandParameters#getSinceId() "since"} revision is specified, it defaults to the
     * parent of the {@link ChangesCommandParameters#getUntilId() "until"} revision. If the "until" revision specifies
     * a merge commit (a commit with multiple parents), <i>no changes will be returned</i>.
     * <p>
     * Note: Some {@link Change changes}, such as {@link ChangeType#COPY copies} and {@link ChangeType#MOVE moves},
     * affect more than one path. When filtering by path, if both paths are not available the change's type may not
     * be correctly detected by some SCMs.
     *
     * @param parameters  parameters describing the revisions to compare and paths to filter by
     * @param pageRequest describes the set of changes to return
     * @return a command which, when executed, will return the requested page of changes
     * @see ChangesCommandParameters
     * @see CommitService#getChanges(ChangesRequest, PageRequest)
     */
    @Nonnull
    Command<Page<Change>> changes(@Nonnull ChangesCommandParameters parameters, @Nonnull PageRequest pageRequest);

    /**
     * Retrieves a page of {@link Changeset changesets} given a set of {@link ChangesetsCommandParameters#getCommitIds
     * commit IDs}, where each changeset includes the first page of {@link Change changes}
     * between a requested commit and its first parent.
     *
     * @param parameters  parameters describing the changesets to retrieve
     * @param pageRequest describes the page of changesets to retrieve
     * @return a command which, when executed, will retrieve a page of changesets
     * @see CommitService#getChangesets(ChangesetsRequest, PageRequest)
     */
    @Nonnull
    Command<Page<Changeset>> changesets(@Nonnull ChangesetsCommandParameters parameters,
                                        @Nonnull PageRequest pageRequest);

    /**
     * Retrieves the common ancestor for the provided commits
     *
     * @param parameters parameters describing which common ancestor to retrieve
     * @return a command which, when executed, returns the common ancestor of the provided commits. If the provided
     *         commits do not have a common ancestor, the command returns {@code null}.
     * @since 5.0
     */
    @Nonnull
    Command<MinimalCommit> commonAncestor(@Nonnull CommonAncestorCommandParameters parameters);

    /**
     * Retrieves {@link Commit details} for the specified commit.
     * <p>
     * If a {@link CommitCommandParameters#getPath() path} is provided, the {@link Changeset} returned <i>may not</i>
     * be for the specified {@link CommitCommandParameters#getCommitId() commit}. Instead, that commit is used as
     * a starting point and details for the first commit in its ancestry which affects that path are returned.
     *
     * @param parameters parameters describing the commit to retrieve details for
     * @return a command which, when executed, will return details for the specified commit
     * @see CommitCommandParameters
     * @see CommitService#getCommit(CommitRequest)
     */
    @Nonnull
    Command<Commit> commit(@Nonnull CommitCommandParameters parameters);

    /**
     * Streams commits which match the provided {@link CommitsCommandParameters parameters} to the provided callback.
     *
     * @param parameters parameters describing the commits to retrieve
     * @param callback   a callback to receive the streamed commits
     * @return a command which, when executed, will stream commits to the provided callback
     * @throws UnsupportedOperationException if a {@link CommitsCommandParameters#getSecondaryRepository() secondary
     *                                       repository} is provided an {@link ScmFeature#CROSS_REPOSITORY cross-
     *                                       repository} operations are not supported
     * @see CommitsCommandParameters
     * @see CommitService#streamCommits(BulkCommitsRequest, CommitCallback)
     * @see CommitService#streamCommits(CommitsRequest, CommitCallback)
     * @see CommitService#streamCommitsBetween(CommitsBetweenRequest, CommitCallback)
     */
    @Nonnull
    Command<Void> commits(@Nonnull CommitsCommandParameters parameters, @Nonnull CommitCallback callback);

    /**
     * Retrieves a {@link Page page} of commits matching the specified {@link CommitsCommandParameters parameters}.
     * <p>
     * If the {@link ScmService#isSupported(Repository, ScmFeature) SCM supports} {@link ScmFeature#CROSS_REPOSITORY
     * cross-repository} operations, a {@link CommitsCommandParameters#getSecondaryRepository() secondary repository}
     * may be provided. Otherwise, it must be omitted or match the target repository.
     *
     * @param parameters  parameters describing which commits to retrieve
     * @param pageRequest describes the set of commits to return
     * @return a command which, when executed, will retrieve a page of commits
     * @throws UnsupportedOperationException if a {@link CommitsCommandParameters#getSecondaryRepository() secondary
     *                                       repository} was provided and the underlying SCM does not support
     *                                       {@link ScmFeature#CROSS_REPOSITORY cross-repository} operations
     * @see CommitsCommandParameters
     * @see CommitService#getCommits(CommitsRequest, PageRequest)
     * @see CommitService#getCommitsBetween(CommitsBetweenRequest, PageRequest)
     */
    @Nonnull
    Command<Page<Commit>> commits(@Nonnull CommitsCommandParameters parameters, @Nonnull PageRequest pageRequest);

    /**
     * Resolves the default branch for the specified repository.
     * <p>
     * The returned command may throw {@link NoDefaultBranchException
     * NoDefaultBranchException}, when executed, if the specified repository does not have a resolvable default branch.
     * This might happen, for example, on a new repository with no commits, or if the only commits to a repository have
     * been to a branch other than the default branch.
     *
     * @return a command which, when executed, will retrieve the repository's default branch
     * @throws NoDefaultBranchException if the repository does not have a resolvable default branch
     * @see RefService#getDefaultBranch(Repository)
     */
    @Nonnull
    Command<Branch> defaultBranch();

    /**
     * Streams a diff between two {@link Commit commits}.
     *
     * @param parameters parameters describing the diff to stream
     * @param callback   the callback to receive the streamed diff
     * @return a command which, when executed, will stream the requested diff
     * @see DiffCommandParameters
     * @see CommitService#streamDiff(DiffRequest, DiffContentCallback)
     */
    @Nonnull
    Command<Void> diff(@Nonnull DiffCommandParameters parameters, @Nonnull DiffContentCallback callback);

    /**
     * Streams the {@link ContentTreeNode tree nodes} (files, subdirectories and submodules) in the specified
     * {@link DirectoryCommandParameters#getPath() directory}.
     *
     * @param parameters  parameters providing the directory to stream and the revision to stream it for
     * @param callback    the callback to receive the streamed nodes
     * @param pageRequest describes the set of nodes to stream
     * @return a command which, when executed, will stream the requested directory's contents
     * @see DirectoryCommandParameters
     * @see ContentService#streamDirectory(Repository, String, String, boolean, ContentTreeCallback, PageRequest)
     */
    @Nonnull
    Command<Void> directory(@Nonnull DirectoryCommandParameters parameters, @Nonnull ContentTreeCallback callback,
                            @Nonnull PageRequest pageRequest);

    /**
     * Streams the contents of the {@link FileCommandParameters specified file}.
     *
     * @param parameters  parameters describing the file to stream and the revision to stream it for
     * @param callback    the callback to receive the streamed content
     * @param pageRequest describes the set of lines in the file to stream
     * @return a command which, when executed, will stream the requested file's content
     * @see FileCommandParameters
     * @see ContentService#streamFile(Repository, String, String, PageRequest, boolean, FileContentCallback)
     */
    @Nonnull
    Command<Void> file(@Nonnull FileCommandParameters parameters, @Nonnull FileContentCallback callback,
                       @Nonnull PageRequest pageRequest);

    /**
     * Streams {@link Ref refs} for all of the heads in the repository. The exact definition of "heads" is an
     * SCM-specific detail, but it will generally include all branches and tags in the repository.
     * <p>
     * <b>Note</b>: This method is not exposed via the service API. It is only available directly on the SCM API.
     *
     * @param callback the ref callback that will be called for each head
     * @return a command which, when executed, will stream repository heads
     */
    @Nonnull
    Command<Void> heads(@Nonnull RefCallback callback);

    /**
     * Streams the <i>raw content</i> of the given file after performing a <i>best effort</i> detection of its
     * MIME type. That MIME type will be used to retrieve an {@code OutputStream} from the provided supplier,
     * and the file's raw content will then be written to the returned stream.
     *
     * @param parameters     parameters used to specify the path and version of the file to stream
     * @param outputSupplier the supplier which, when given a content type, will provide an output stream
     * @return a command which, when executed, will determine the file's type and then stream its data
     * @see RawFileCommandParameters
     * @see ContentService#streamFile(Repository, String, String, TypeAwareOutputSupplier)
     */
    @Nonnull
    Command<Void> rawFile(@Nonnull RawFileCommandParameters parameters,
                          @Nonnull TypeAwareOutputSupplier outputSupplier);

    /**
     * Resolves the specified {@link ResolveRefCommandParameters#getRefId refId}, which may be a:
     * <ul>
     *     <li>{@link Branch#getId() branch name}</li>
     *     <li>{@link Tag#getId() tag name}</li>
     *     <li>{@link Commit#getId() commit hash}</li>
     * </ul>
     * If the {@code refId} should be resolved to a specific type, that {@link ResolveRefCommandParameters#getType
     * type} should be set to allow the SCM to restrict its search.
     * <p>
     * If a hash is provided, it will be resolved to the branch or tag it is the tip of. If it is not the tip of
     * any branch or tag, it will resolve to {@code null}. If a {@link ResolveRefCommandParameters#getType type}
     * was specified, the hash will only be resolved against that type.
     * <p>
     * When a hash is provided and resolves to multiple branches, tags or a combination of both, it is left to the
     * SCM implementation to determine the result. SCM implementors are <i>encouraged</i> to choose a tag over a
     * branch, and to return the "first" branch or tag, ordered alphabetically, but this is not enforced.
     *
     * @param parameters parameters describing the ref to resolve
     * @return a command which, when executed, will resolve the specified ref
     * @see ResolveRefCommandParameters
     * @see RefService#resolveRef(ResolveRefRequest)
     * @since 4.6
     */
    @Nonnull
    Command<Ref> resolveRef(@Nonnull ResolveRefCommandParameters parameters);

    /**
     * Resolves refs from the ref IDs provided on {@code parameters}, which may be one of
     * <ul>
     *     <li>{@link Branch#getId() branch name}</li>
     *     <li>{@link Tag#getId() tag name}</li>
     *     <li>{@link Commit#getId() commit hash}</li>
     * </ul>
     * Callers can specify on the {@code parameters} whether a provided ID is expected to resolve to a
     * {@link ResolveRefsCommandParameters#getBranchIds() branch}, {@link ResolveRefsCommandParameters#getTagIds() tag},
     * or a {@link ResolveRefsCommandParameters#getRefIds() ref of unknown type}.
     * <p>
     * If hashes are provided, they will be resolved to branches or tags they are the tip of. If they are not the tip of
     * any branch or tag, they will resolve to {@code null} and will not be included in the returned map.
     * <p>
     * When a hash is provided and resolves to multiple branches, tags or a combination of both, it is left to the
     * SCM implementation to determine the result. SCM implementors are <i>encouraged</i> to choose a tag over a
     * branch, and to return the "first" branch or tag, ordered alphabetically, but this is not enforced.
     *
     * @param parameters describes the ref IDs to resolve
     * @return a command which, when executed, will resolve the specified refs
     * @see ResolveRefsCommandParameters
     * @see RefService#resolveRefs(ResolveRefsRequest)
     * @since 4.4
     */
    @Nonnull
    Command<Map<String, Ref>> resolveRefs(@Nonnull ResolveRefsCommandParameters parameters);

    /**
     * Streams {@link Tag tags} ordered either alphabetically or by modification and optionally filtered.
     * <p>
     * When no explicit order is requested, the default ordering depends on whether a filter was provided:
     * <ul>
     *     <li>If filtering, tags default to {@link RefOrder#ALPHABETICAL alphabetical} order</li>
     *     <li>Otherwise, tags default to {@link RefOrder#MODIFICATION modification} order</li>
     * </ul>
     * Modification order retrieves tags based on the date of their most recent commit.
     *
     * @param parameters parameters describing desired order and filter text
     * @param callback   a callback to receive the streamed commits
     * @return a command which, when executed, will stream tags to the provided callback
     * @see TagsCommandParameters
     * @see RefService#streamTags(RepositoryTagsRequest, TagCallback)
     */
    @Nonnull
    Command<Void> tags(@Nonnull TagsCommandParameters parameters, @Nonnull TagCallback callback);

    /**
     * Retrieves a {@link Page page} of {@link Tag tags} ordered either alphabetically or by modification and
     * optionally filtered.
     * <p>
     * When no explicit order is requested, the default ordering depends on whether a filter was provided:
     * <ul>
     *     <li>If filtering, tags default to {@link RefOrder#ALPHABETICAL alphabetical} order</li>
     *     <li>Otherwise, tags default to {@link RefOrder#MODIFICATION modification} order</li>
     * </ul>
     * Modification order retrieves tags based on the date of their most recent commit.
     *
     * @param parameters  parameters describing desired order and filter text
     * @param pageRequest describes the set of tags to return
     * @return a command which, when executed, will return the requested page of tags
     * @see TagsCommandParameters
     * @see RefService#getTags(RepositoryTagsRequest, PageRequest)
     */
    @Nonnull
    Command<Page<Tag>> tags(@Nonnull TagsCommandParameters parameters, @Nonnull PageRequest pageRequest);

    /**
     * Streams all of the commits in the repository reachable from any branch or tag, traversed in topological order.
     *
     * @param callback the callback to receive the streamed commits
     * @return a command which, when executed, will stream a topological traversal of all of the commits in the
     *         repository
     * @see CommitService#traverse(TraversalRequest, TraversalCallback)
     */
    @Nonnull
    Command<Void> traverseCommits(@Nonnull TraversalCallback callback);

    /**
     * Retrieves the {@link ContentTreeNode.Type type} for the specified {@link TypeCommandParameters#getPath() path}
     * at the requested revision, if the path is present. Note that the same path might have different types between
     * revisions but the root entry ("", "\" or "/") will always be a {@link ContentTreeNode.Type#DIRECTORY directory}.
     *
     * @param parameters parameters describing the path to check, and the revision to check it for
     * @return a command which, when executed, will return the requested path's type
     * @see TypeCommandParameters
     * @see ContentService#getType(Repository, String, String)
     */
    @Nonnull
    Command<ContentTreeNode.Type> type(@Nonnull TypeCommandParameters parameters);
}
