package com.atlassian.bitbucket.scm;

import com.atlassian.bitbucket.content.ContentTreeContext;
import com.atlassian.bitbucket.content.File;
import org.apache.commons.lang3.StringUtils;

import javax.annotation.Nonnull;

public class DirectoryCommandParameters extends AbstractCommitCommandParameters {

    private final boolean recursive;
    private final boolean withSizes;

    private DirectoryCommandParameters(Builder builder) {
        super(builder);

        if (StringUtils.isBlank(getCommitId())) {
            throw new IllegalStateException("A commit ID must be provided when retrieving directory contents");
        }
        recursive = builder.recursive;
        withSizes = builder.withSizes;
    }

    public boolean isRecursive() {
        return recursive;
    }

    /**
     * @return {@code true} if {@link File#getSize() file sizes} should be populated; otherwise, {@code false}
     * @since 4.2
     */
    public boolean isWithSizes() {
        return withSizes;
    }

    /**
     * Assembles a {@link ContentTreeContext} from these parameters.
     *
     * @return a {@link ContentTreeContext} describing these parameters
     */
    @Nonnull
    public ContentTreeContext toContext() {
        return new ContentTreeContext.Builder(getCommitId())
                .recursive(isRecursive())
                .build();
    }

    public static class Builder extends AbstractCommitParametersBuilder<Builder> {

        private boolean recursive;
        private boolean withSizes;

        public Builder() {
        }

        @Nonnull
        public DirectoryCommandParameters build() {
            return new DirectoryCommandParameters(this);
        }

        @Nonnull
        public Builder recurse(boolean value) {
            recursive = value;

            return self();
        }

        /**
         * @param value {@code true} to populate {@link File#getSize() file sizes}; otherwise, {@code false} to
         *              leave them empty
         * @return {@code this}
         */
        @Nonnull
        public Builder withSizes(boolean value) {
            withSizes = value;

            return self();
        }

        @Nonnull
        @Override
        protected Builder self() {
            return this;
        }
    }
}
