package com.atlassian.bitbucket.scm;

import com.atlassian.bitbucket.content.DiffContext;
import org.apache.commons.lang3.StringUtils;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

public class DiffCommandParameters extends AbstractDiffCommandParameters {

    private final String sinceId;
    private final String untilId;

    private DiffCommandParameters(Builder builder) {
        super(builder);

        sinceId = builder.sinceId;
        untilId = builder.untilId;
    }

    @Nullable
    public String getSinceId() {
        return sinceId;
    }

    @Nonnull
    public String getUntilId() {
        return untilId;
    }

    /**
     * Assembles a {@link DiffContext} from these parameters.
     *
     * @return a {@link DiffContext} describing these parameters
     */
    @Nonnull
    public DiffContext toContext() {
        return toContext(new DiffContext.Builder(getSinceId(), getUntilId()));
    }

    public static class Builder extends AbstractBuilder<Builder> {

        private String sinceId;
        private String untilId;

        public Builder() {
        }

        @Nonnull
        public DiffCommandParameters build() {
            if (StringUtils.isBlank(untilId)) {
                throw new IllegalStateException("An \"until\" ID is required to build a diff");
            }

            return new DiffCommandParameters(this);
        }

        @Nonnull
        public Builder sinceId(@Nullable String value) {
            sinceId = value;

            return this;
        }

        @Nonnull
        public Builder untilId(@Nonnull String value) {
            //Nullability is enforced in build(); "value" is annotated @Nonnull to help callers avoid errors
            untilId = value;

            return this;
        }

        @Nonnull
        @Override
        protected Builder self() {
            return this;
        }
    }
}
