package com.atlassian.bitbucket.scm;

import com.atlassian.bitbucket.commit.*;
import com.atlassian.bitbucket.repository.Repository;
import com.google.common.collect.ImmutableSet;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.Collections;
import java.util.Set;

import static com.atlassian.bitbucket.commit.AbstractCommitsRequest.UNLIMITED_MESSAGE_LENGTH;
import static java.util.Objects.requireNonNull;

public class CommitsCommandParameters extends AbstractCommandParameters {

    private final boolean all;
    private final Set<String> excludes;
    private final boolean followRenames;
    private final boolean ignoringMissing;
    private final Set<String> includes;
    private final int maxMessageLength;
    private final CommitListMergeFilter merges;
    private final CommitOrder order;
    private final Set<String> paths;
    private final Repository secondaryRepository;
    private final boolean traversing;
    private final boolean withMessages;

    private CommitsCommandParameters(Builder builder) {
        this.all = builder.all;
        //If a traversal was not requested, ignore any excludes
        this.excludes = builder.traversing ? builder.excludes.build() : Collections.emptySet();
        this.followRenames = builder.followRenames;
        this.ignoringMissing = builder.ignoringMissing;
        //If "all" was specified, ignore any explicit includes
        this.includes = all ? Collections.emptySet() : builder.includes.build();
        //If builder.withMessages is true, normalize the requested length. Otherwise, ignore it. This means the
        //length can either be 0 because it was set that way explicitly, or because withMessages was false
        maxMessageLength = builder.withMessages ? Math.max(UNLIMITED_MESSAGE_LENGTH, builder.maxMessageLength) : 0;
        this.merges = builder.merges;
        this.order = builder.order;
        this.paths = builder.paths.build();
        this.secondaryRepository = builder.secondaryRepository;
        this.traversing = builder.traversing;
        //Only set withMessages to true if builder.withMessages was true _and_ the max length isn't 0
        this.withMessages = builder.withMessages && maxMessageLength != 0;

        if (followRenames) {
            if (paths.isEmpty()) {
                throw new IllegalStateException("A path must be supplied when attempting to follow renames");
            }
            if (paths.size() > 1) {
                throw new IllegalStateException("Following renames is only possible with a single path");
            }
        }
    }

    @Nonnull
    public Set<String> getExcludes() {
        return excludes;
    }

    @Nonnull
    public Set<String> getIncludes() {
        return includes;
    }

    /**
     * @return the maximum length to read for {@link Commit#getMessage() commit messages}; if set, messages
     *         longer than this will be truncated; -1 indicates unlimited message length
     * @since 4.5
     */
    public int getMaxMessageLength() {
        return maxMessageLength;
    }

    /**
     * Retrieves the strategy being used to filter merge commits.
     *
     * @return how merge commits will be filtered
     * @since 4.8
     * @see CommitListMergeFilter
     */
    @Nonnull
    public CommitListMergeFilter getMerges() {
        return merges;
    }

    /**
     * Retrieves the sort order to use for the commits.
     *
     * @return the sort order
     * @since 5.0
     */
    @Nonnull
    public CommitOrder getOrder() {
        return order;
    }

    @Nonnull
    public Set<String> getPaths() {
        return paths;
    }

    @Nonnull
    public Repository getSecondaryRepository() {
        return secondaryRepository;
    }

    public boolean hasExcludes() {
        return !excludes.isEmpty();
    }

    public boolean hasIncludes() {
        return !includes.isEmpty();
    }

    public boolean hasPaths() {
        return !paths.isEmpty();
    }

    public boolean hasSecondaryRepository() {
        return secondaryRepository != null;
    }

    public boolean isAll() {
        return all;
    }

    /**
     * Whether or not the commit history will attempt to follow renames. This is only applicable for individual file
     * {@link #getPaths() paths}
     *
     * @return {@code true} if renames are followed, {@code false} otherwise
     *
     * @since 4.5
     */
    public boolean isFollowingRenames() {
        return followRenames;
    }

    /**
     * @return whether or not to ignore missing commits.
     *
     * @since 5.0
     */
    public boolean isIgnoringMissing() {
        return ignoringMissing;
    }

    public boolean isTraversing() {
        return traversing;
    }

    /**
     * @return {@code true} if the comment messages should be retrieved. {@code false} otherwise.
     */
    public boolean isWithMessages() {
        return withMessages;
    }

    /**
     * Assembles a {@link CommitContext} from these parameters.
     *
     * @return a {@link CommitContext} describing these parameters
     */
    @Nonnull
    public CommitContext toContext() {
        return new CommitContext.Builder().build();
    }

    public static class Builder {

        private final ImmutableSet.Builder<String> excludes;
        private final ImmutableSet.Builder<String> includes;
        private final ImmutableSet.Builder<String> paths;

        private boolean all;
        private boolean followRenames;
        private boolean ignoringMissing;
        private int maxMessageLength;
        private CommitListMergeFilter merges;
        private CommitOrder order;
        private Repository secondaryRepository;
        private boolean traversing;
        private boolean withMessages;

        public Builder() {
            excludes = ImmutableSet.builder();
            includes = ImmutableSet.builder();
            maxMessageLength = UNLIMITED_MESSAGE_LENGTH;
            merges = CommitListMergeFilter.INCLUDE;
            order = CommitOrder.DEFAULT;
            paths = ImmutableSet.builder();
            traversing = true;
            withMessages = true;
        }

        @SuppressWarnings("deprecation")
        public Builder(@Nonnull CommitsBetweenRequest request) {
            this();

            exclude(requireNonNull(request, "request").getExcludes())
                    .followRenames(request.isFollowingRenames())
                    .ignoreMissing(request.isIgnoringMissing())
                    .include(request.getIncludes())
                    .maxMessageLength(request.getMaxMessageLength())
                    .merges(request.getMerges())
                    .order(request.getOrder())
                    .paths(request.getPaths())
                    .secondaryRepository(request.getSecondaryRepository());
        }

        @SuppressWarnings("deprecation")
        public Builder(@Nonnull CommitsRequest request) {
            this();

            followRenames(requireNonNull(request, "request").isFollowingRenames())
                    .ignoreMissing(request.isIgnoringMissing())
                    .include(request.getCommitId())
                    .maxMessageLength(request.getMaxMessageLength())
                    .merges(request.getMerges())
                    .order(request.getOrder())
                    .paths(request.getPaths());
        }

        @Nonnull
        public Builder all(boolean value) {
            all = value;

            return this;
        }

        @Nonnull
        public CommitsCommandParameters build() {
            return new CommitsCommandParameters(this);
        }

        @Nonnull
        public Builder exclude(@Nullable Iterable<String> values) {
            addIf(NOT_BLANK, excludes, values);

            return this;
        }

        @Nonnull
        public Builder exclude(@Nullable String value, @Nullable String... values) {
            addIf(NOT_BLANK, excludes, value, values);

            return this;
        }

        /**
         * @since 4.5
         */
        @Nonnull
        public Builder followRenames(boolean value) {
            followRenames = value;

            return this;
        }

        /**
         * @since 5.0
         */
        @Nonnull
        public Builder ignoreMissing(boolean value) {
            ignoringMissing = value;

            return this;
        }

        @Nonnull
        public Builder include(@Nullable Iterable<String> values) {
            addIf(NOT_BLANK, includes, values);

            return this;
        }

        @Nonnull
        public Builder include(@Nullable String value, @Nullable String... values) {
            addIf(NOT_BLANK, includes, value, values);

            return this;
        }

        /**
         * @since 4.5
         */
        @Nonnull
        public Builder maxMessageLength(int value) {
            maxMessageLength = value;

            return this;
        }

        /**
         * @since 4.8
         */
        @Nonnull
        public Builder merges(@Nonnull CommitListMergeFilter value) {
            merges = requireNonNull(value, "merges");

            return this;
        }

        /**
         * @since 5.0
         */
        @Nonnull
        public Builder order(@Nonnull CommitOrder value) {
            order = requireNonNull(value, "order");

            return this;
        }

        @Nonnull
        public Builder path(@Nullable String value) {
            addIf(NOT_BLANK, paths, value);

            return this;
        }

        @Nonnull
        public Builder paths(@Nullable Iterable<String> values) {
            addIf(NOT_BLANK, paths, values);

            return this;
        }

        @Nonnull
        public Builder paths(@Nullable String value, @Nullable String... values) {
            addIf(NOT_BLANK, paths, value, values);

            return this;
        }

        @Nonnull
        public Builder secondaryRepository(@Nullable Repository value) {
            secondaryRepository = value;

            return this;
        }

        @Nonnull
        public Builder traverse(boolean value) {
            traversing = value;

            return this;
        }

        /**
         * @param value if commit messages should be included in the command output.
         * @return the updated builder.
         */
        @Nonnull
        public Builder withMessages(boolean value) {
            withMessages = value;

            return this;
        }
    }
}
