package com.atlassian.bitbucket.scm;

import com.atlassian.bitbucket.commit.Commit;
import com.atlassian.bitbucket.commit.CommitRequest;
import org.apache.commons.lang3.StringUtils;

import javax.annotation.Nonnull;

import static com.atlassian.bitbucket.commit.AbstractCommitRequest.UNLIMITED_MESSAGE_LENGTH;

public class CommitCommandParameters extends AbstractCommitCommandParameters {

    private final int maxMessageLength;

    private CommitCommandParameters(Builder builder) {
        super(builder);

        if (StringUtils.isBlank(getCommitId())) {
            throw new IllegalStateException("A commit ID is required to retrieve a commit");
        }

        //Normalize negative maxMessageLengths to -1, and preserve any other value as-is
        maxMessageLength = Math.max(UNLIMITED_MESSAGE_LENGTH, builder.maxMessageLength);
    }

    /**
     * @return the maximum length to read for the {@link Commit#getMessage() commit message}, which may be {@code 0}
     *         to omit the message entirely or -1 to include the full message regardless of length
     * @since 4.5
     */
    public int getMaxMessageLength() {
        return maxMessageLength;
    }

    public static class Builder extends AbstractCommitParametersBuilder<Builder> {

        private int maxMessageLength;

        public Builder() {
            maxMessageLength = UNLIMITED_MESSAGE_LENGTH;
        }

        public Builder(@Nonnull CommitRequest request) {
            super(request);

            maxMessageLength = request.getMaxMessageLength();
        }

        @Nonnull
        public CommitCommandParameters build() {
            return new CommitCommandParameters(this);
        }

        /**
         * @since 4.5
         */
        @Nonnull
        public Builder maxMessageLength(int value) {
            maxMessageLength = value;

            return self();
        }

        @Nonnull
        @Override
        protected Builder self() {
            return this;
        }
    }
}
