package com.atlassian.bitbucket.repository;

import javax.annotation.Nonnull;
import java.util.Objects;

import static java.util.Objects.requireNonNull;

/**
 * A simple, immutable implementation of {@link RefChange}. To enhance compatibility across releases, instances must
 * be created using the nested {@link Builder builder}.
 */
public class SimpleRefChange implements RefChange {

    private final MinimalRef ref;
    private final String fromHash;
    private final String toHash;
    private final RefChangeType type;

    /**
     * @since 5.10
     */
    protected SimpleRefChange(@Nonnull AbstractBuilder<?> builder) {
        this.fromHash = builder.fromHash;
        this.ref = builder.ref;
        this.toHash = builder.toHash;
        this.type = builder.type;
    }

    /**
     * @deprecated in 5.10 for removal in 6.0. Use {@link SimpleRefChange.Builder} instead.
     */
    @Deprecated
    protected SimpleRefChange(@Nonnull MinimalRef ref, @Nonnull String fromHash, @Nonnull String toHash,
                              @Nonnull RefChangeType type) {
        this.fromHash = fromHash;
        this.ref = ref;
        this.toHash = toHash;
        this.type = type;
    }

    @Nonnull
    @Override
    public String getFromHash() {
        return fromHash;
    }

    @Nonnull
    @Override
    public MinimalRef getRef() {
        return ref;
    }

    @Nonnull
    @Override
    public String getToHash() {
        return toHash;
    }

    @Nonnull
    @Override
    public RefChangeType getType() {
        return type;
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o instanceof SimpleRefChange) {
            SimpleRefChange that = (SimpleRefChange) o;

            return fromHash.equals(that.fromHash) &&
                    Objects.equals(ref.getId(), that.ref.getId()) &&
                    Objects.equals(ref.getType(), that.ref.getType()) &&
                    toHash.equals(that.toHash) &&
                    type.equals(that.type);
        }

        return false;
    }

    public int hashCode() {
        int result = ref.getId().hashCode();
        result = 31 * result + fromHash.hashCode();
        result = 31 * result + toHash.hashCode();
        result = 31 * result + type.hashCode();
        return result;
    }

    @Override
    public String toString() {
        return getRef().getId() + ": " + getFromHash() + " -> " + getToHash() + " (" + getType() + ")";
    }

    public static class Builder extends AbstractBuilder<SimpleRefChange.Builder> {

        public Builder() {
        }

        public Builder(@Nonnull RefChange refChange) {
            super(refChange);
        }

        @Nonnull
        public SimpleRefChange build() {
            validate();

            return new SimpleRefChange(self());
        }

        @Nonnull
        public Builder from(@Nonnull Ref value) {
            return super.from(value);
        }

        @Nonnull
        public Builder fromHash(@Nonnull String value) {
            return super.fromHash(value);
        }

        @Nonnull
        public Builder ref(@Nonnull MinimalRef value) {
            return super.ref(value);
        }

        @Override
        public Builder self() {
            return this;
        }

        @Nonnull
        public Builder to(@Nonnull Ref value) {
            return super.to(value);
        }

        @Nonnull
        public Builder toHash(@Nonnull String value) {
            return super.toHash(value);
        }

        @Nonnull
        public Builder type(@Nonnull RefChangeType value) {
            return super.type(value);
        }
    }

    /**
     * @since 5.10
     */
    public abstract static class AbstractBuilder<B extends AbstractBuilder<B>> {

        private String fromHash;
        private MinimalRef ref;
        private String toHash;
        private RefChangeType type;

        protected AbstractBuilder() {
        }

        public AbstractBuilder(@Nonnull RefChange change) {
            fromHash = requireNonNull(change, "change").getFromHash();
            ref = change.getRef();
            toHash = change.getToHash();
            type = change.getType();
        }

        @Nonnull
        public B from(@Nonnull Ref value) {
            fromHash = requireNonNull(value, "ref").getLatestCommit();

            return ref(value);
        }

        @Nonnull
        public B fromHash(@Nonnull String value) {
            fromHash = requireNonNull(value, "value");

            return self();
        }

        @Nonnull
        public B ref(@Nonnull MinimalRef value) {
            ref = requireNonNull(value, "value");

            return self();
        }

        @Nonnull
        public B to(@Nonnull Ref value) {
            toHash = requireNonNull(value, "ref").getLatestCommit();

            return ref(value);
        }

        @Nonnull
        public B toHash(@Nonnull String value) {
            toHash = requireNonNull(value, "value");

            return self();
        }

        @Nonnull
        public B type(@Nonnull RefChangeType value) {
            type = requireNonNull(value, "value");

            return self();
        }

        protected void validate() {
            requireNonNull(fromHash, "fromHash");
            requireNonNull(ref, "ref");
            requireNonNull(toHash, "toHash");
            requireNonNull(type, "type");
        }

        protected abstract B self();
    }
}
