package com.atlassian.bitbucket.repository;

import com.atlassian.bitbucket.project.PersonalProject;
import com.atlassian.bitbucket.project.Project;
import org.apache.commons.lang3.StringUtils;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.base.Preconditions.checkState;

/**
 * Defines the properties that can be set when forking a {@link Repository}. To reduce backwards compatibility issues
 * as new properties are added over time, instances of this class may only be created using its {@link Builder}.
 * <p>
 * The following property is <i>required</i>:
 * <ul>
 *     <li>{@link #getParent()}: The repository to fork</li>
 * </ul>
 * The following properties, if not specified, will have defaults applied:
 * <ul>
 *     <li>{@link #getName()}: Defaults to the {@link #getParent() parent} repository's {@link Repository#getName()
 *     name} if not explicitly specified</li>
 *     <li>{@link #getProject()}: Defaults to the <i>forking user's</i> {@link PersonalProject personal project}
 *     if not explicitly specified</li>
 *     <li>{@link #isForkable()}: Defaults to {@code true}, meaning the fork will itself be forkable, if not
 *     explicitly specified</li>
 *     <li>{@link #isPublic()}: Defaults to {@code false}, meaning the fork will will not be public, if not
 *     explicitly specified</li>
 * </ul>
 * The fork's {@link Repository#getSlug() slug}, which is used in URLs (both for the browser and when cloning), will be
 * generated from the provided {@link #getName()}. Both the name and the generated slug must be unique within the
 * {@link #getProject() project} or the fork cannot be created.
 *
 * @see RepositoryService#fork(RepositoryForkRequest)
 */
public class RepositoryForkRequest extends AbstractRepositoryRequest {

    private final Repository parent;
    private final Project project;

    private RepositoryForkRequest(String name, boolean forkable, Repository parent, Project project, boolean publiclyAccessible) {
        super(name, forkable, publiclyAccessible);

        this.parent = parent;
        this.project = project;
    }

    /**
     * Retrieves the repository to be forked. This will become the {@link Repository#getOrigin() origin} of the new
     * fork.
     *
     * @return the repository to be forked
     */
    @Nonnull
    public Repository getParent() {
        return parent;
    }

    /**
     * Retrieves the project in which the fork should be created, or {@code null} to create the fork in the <i>current
     * user's</i> {@link PersonalProject personal project}.
     *
     * @return the destination repository, or {@code null} to use the current user's personal project
     */
    @Nullable
    public Project getProject() {
        return project;
    }

    /**
     * Constructs {@link RepositoryForkRequest requests} for forking repositories.
     */
    public static class Builder extends AbstractBuilder<Builder> {

        private Repository parent;
        private Project project;

        public Builder() {
        }

        /**
         * Constructs a new {@code Builder} which will use the specified {@link Repository} as the fork's
         * {@link RepositoryForkRequest#getParent() parent}. The parent's {@link Repository#getName() name}
         * is set as the fork's name, but the parent's {@link Repository#getProject() project} is <i>not</i>
         * set as the fork's project. Since names and slugs must be unique within a project, defaulting both
         * the name and project would produce a request guaranteed to fail unless one property or the other
         * was changed before {@link #build() building}.
         *
         * @param repository the {@link RepositoryForkRequest#getParent() parent} repository for the new fork
         */
        public Builder(@Nonnull Repository repository) {
            super(repository);

            //While this _could_ copy the repository's project, it doesn't make sense to copy the name _and_ the
            //project; that combination would require the assembler to replace one or the other before building
            //or it would result in a constraint violation. Instead, the project is ignored (making the default
            //target the forking user's personal project).
            parent = repository;

            // Always default to private unless overridden
            publiclyAccessible = false;
        }

        /**
         * Constructs a new {@code Builder} which uses as its defaults all the values from the provided request.
         *
         * @param request the request to copy into the new builder
         */
        public Builder(@Nonnull RepositoryForkRequest request) {
            super(request);

            parent = request.getParent();
            project = request.getProject();
        }

        /**
         * Builds a {@link RepositoryForkRequest request} from the assembled values. Before the request is built,
         * it is first verified that a {@link #parent(Repository) parent} was specified and an exception is thrown
         * if it was not.
         * <p>
         * If no {@link #name(String) name} was explicitly set, the {@link #parent(Repository) parent} repository's
         * {@link Repository#getName() name} will be set as the fork's name. This ensures the constructed request
         * honours its nullability contract.
         *
         * @return the built request
         * @throws IllegalStateException if the {@link #parent(Repository) parent} is {@code null}
         */
        @Nonnull
        public RepositoryForkRequest build() {
            //The name will be defaulted to the parent's name, if it is null, so that is not checked. The project
            //will default to the forking user's personal project, if it is null, so that is also not checked.
            checkState(parent != null, "parent");

            //To honour the contract of AbstractRepositoryRequest.getName(), default the name here rather than
            //leaving it null. The repository service would do the same
            return new RepositoryForkRequest(StringUtils.defaultIfBlank(name, parent.getName()),
                    forkable, parent, project, publiclyAccessible);
        }

        //Overrides the nullability on the AbstractBuilder. When creating a fork, the name will be inferred from the
        //parent repository if it is not provided
        /**
         * Sets the name to be used for the request.
         * <p>
         * If this value is {@code null}, empty or contains only whitespace, or this method is not called prior to
         * {@link #build() building}, the {@link #parent(Repository) parent's} {@link Repository#getName() name}
         * will be used by default.
         *
         * @param value the name for the repository, which may be blank to inherit the parent's name
         * @return {@code this}
         */
        @Nonnull
        @Override
        public Builder name(@Nullable String value) {
            name = value;

            return self();
        }

        /**
         * Sets the repository to be forked, which will be the {@link Repository#getOrigin() origin} of the fork.
         * <p>
         * Note: This value is <i>required</i>. If this method is not called prior to {@link #build() building} the
         * request, an exception will be thrown.
         *
         * @param value the repository to fork
         * @return {@code this}
         * @throws NullPointerException if the provided {@code value} is {@code null}
         */
        @Nonnull
        public Builder parent(@Nonnull Repository value) {
            parent = checkNotNull(value, "parent");

            return self();
        }

        /**
         * Sets the project to which the {@link #parent(Repository) parent} will be forked.
         * <p>
         * If this value is not set, or this method is not called prior to {@link #build() building}, the <i>current
         * user's</i> {@link PersonalProject personal project} will be used by default.
         *
         * @param value the destination project, which may be {@code null} to use the current user's personal project
         * @return {@code this}
         */
        @Nonnull
        public Builder project(@Nullable Project value) {
            project = value;

            return self();
        }

        /**
         * @return {@code this}
         */
        @Override
        protected Builder self() {
            return this;
        }
    }
}
