package com.atlassian.bitbucket.repository;

import javax.annotation.Nonnull;
import java.io.IOException;

/**
 * A callback for streaming {@link Ref refs}.
 *
 * @see AbstractRefCallback
 */
@FunctionalInterface
public interface RefCallback {

    /**
     * Called after the final {@link #onRef(Ref) ref} has been streamed.
     * <p>
     * Note: If there were no refs, this method may be called immediately after {@link #onStart(RefContext)} without
     * any calls to {@link #onRef(Ref)}.
     *
     * @param summary summarizes the request and the streamed refs
     * @throws IOException May be thrown by implementations which perform I/O.
     */
    default void onEnd(@Nonnull RefSummary summary) throws IOException {
    }

    /**
     * Called to process a {@link Ref ref}.
     *
     * @param ref the {@link Ref ref} to stream
     * @return {@code true} if further refs should be streamed; otherwise, {@code false} to stop streaming
     * @throws IOException May be thrown by implementations which perform I/O.
     */
    boolean onRef(@Nonnull Ref ref) throws IOException;

    /**
     * Called before the <i>first</i> {@link #onRef(Ref) ref} is streamed.
     *
     * @param context provides details about the request for which refs are being streamed
     * @throws IOException May be thrown by implementations which perform I/O.
     */
    default void onStart(@Nonnull RefContext context) throws IOException {
    }
}
