package com.atlassian.bitbucket.pull;

import com.atlassian.bitbucket.property.PropertyMap;
import com.atlassian.bitbucket.util.BuilderSupport;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSet;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.Collection;
import java.util.Date;
import java.util.Objects;

/**
 * Request for searching for pull requests. The purpose and behaviour of each field is described on its accessor.
 */
public class PullRequestSearchRequest {

    public static final int MAXIMUM_BRANCHES_TO_SEARCH = 100;

    private final Date closedSince;
    private final Collection<String> fromRefIds;
    private final Integer fromRepositoryId;
    private final PullRequestOrder order;
    private final Collection<PullRequestParticipantRequest> participants;
    private final PullRequestState state;
    private final Collection<String> toRefIds;
    private final Integer toRepositoryId;
    private final boolean withProperties;

    private PullRequestSearchRequest(Builder builder) {
        this.closedSince = builder.closedSince;
        this.fromRefIds = builder.fromRefIds.build();
        this.fromRepositoryId = builder.fromRepositoryId;
        this.order = builder.order;
        this.participants = builder.participants.build();
        this.state = builder.state;
        this.toRefIds = builder.toRefIds.build();
        this.toRepositoryId = builder.toRepositoryId;
        this.withProperties = builder.withProperties;

        if ((fromRefIds.size() + toRefIds.size()) > MAXIMUM_BRANCHES_TO_SEARCH) {
            throw new IllegalArgumentException(getClass().getSimpleName() +
                    " does not allow searching more than " + MAXIMUM_BRANCHES_TO_SEARCH + " branches");
        }
    }

    /**
     * When set, limits the pull requests to search for pull requests in the closed state (i.e.
     * {@link PullRequestState#MERGED} or {@link PullRequestState#DECLINED}) where the closed timestamp is
     * more recent than the returned date.
     *
     * @return the closed since date
     * @since 4.10
     */
    @Nullable
    public Date getClosedSince() {
        return closedSince;
    }

    /**
     * When set, limits the pull requests to search for by the branches the changes are coming from.
     *
     * @return the set branch ids
     */
    @Nonnull
    public Collection<String> getFromRefIds() {
        return fromRefIds;
    }

    /**
     * When set, limits the pull requests to search for by the repository the changes are coming from.
     *
     * @return the set from repository ID
     */
    @Nullable
    public Integer getFromRepositoryId() {
        return fromRepositoryId;
    }

    /**
     * The {@link PullRequestOrder order} to return the search results in. A {@code null} value will be interpreted
     * as {@link PullRequestOrder#NEWEST NEWEST first}).
     *
     * @return required order
     */
    @Nullable
    public PullRequestOrder getOrder() {
        return order;
    }

    /**
     * When set, limits the pull requests to search for by the {@link PullRequestParticipantRequest participants}
     * associated with it.
     *
     * @return The required pull request participants.
     */
    @Nonnull
    public Collection<PullRequestParticipantRequest> getParticipants() {
        return participants;
    }

    /**
     * Restricts the search to a specified {@link PullRequestState pull request state}.
     *
     * @return the state to restrict to
     */
    @Nullable
    public PullRequestState getState() {
        return state;
    }

    /**
     * When set, limits the pull requests to search for by the branches the changes are going to.
     *
     * @return the set branch ids
     */
    @Nonnull
    public Collection<String> getToRefIds() {
        return toRefIds;
    }

    /**
     * When set, limits the pull requests to search for by the repository the changes are going to.
     *
     * @return the set to repository id
     */
    @Nullable
    public Integer getToRepositoryId() {
        return toRepositoryId;
    }

    /**
     * @return {@code true} to fetch the pull requests' properties.
     */
    public boolean isWithProperties() {
        return withProperties;
    }

    public static class Builder extends BuilderSupport {

        private final ImmutableSet.Builder<String> fromRefIds;
        private final ImmutableList.Builder<PullRequestParticipantRequest> participants;
        private final ImmutableSet.Builder<String> toRefIds;

        private Date closedSince;
        private Integer fromRepositoryId;
        private PullRequestOrder order;
        private PullRequestState state;
        private Integer toRepositoryId;
        private boolean withProperties;

        public Builder() {
            fromRefIds = ImmutableSet.builder();
            participants = ImmutableList.builder();
            toRefIds = ImmutableSet.builder();
            withProperties = true;
        }

        /**
         * Assembles a new {@link PullRequestSearchRequest} from the provided values.
         *
         * @return a new pull request search request instance
         */
        @Nonnull
        public PullRequestSearchRequest build() {
            return new PullRequestSearchRequest(this);
        }

        /**
         * When set, limits the pull requests to search for pull requests in the closed state (i.e.
         * {@link PullRequestState#MERGED} or {@link PullRequestState#DECLINED}) where the closed timestamp is
         * more recent than the specified date.
         *
         * @param value the closed since date
         * @return the current builder
         * @since 4.10
         */
        @Nonnull
        public Builder closedSince(@Nullable Date value) {
            closedSince = value;
            return this;
        }

        /**
         * Sets the repository and branches fields based on direction, added here as the use case is so common.
         * <p>
         * If {@link PullRequestDirection#OUTGOING OUTGOING} is the direction the from repository and branch are set
         * otherwise the to repository and branch are set.
         *
         * @param direction  the pull request direction
         * @param repositoryId the repository to restrict by
         * @param branch     the branch to restrict by
         * @return the current builder
         * @throws IllegalStateException if the repositoryId has already been set to a different value
         */
        @Nonnull
        public Builder repositoryAndBranch(@Nullable PullRequestDirection direction, @Nullable Integer repositoryId,
                                           @Nullable String branch) {
            if (direction == PullRequestDirection.OUTGOING) {
                addIf(NOT_BLANK, fromRefIds, branch);
                if (fromRepositoryId != null && !fromRepositoryId.equals(repositoryId)) {
                    throw new IllegalStateException("Repository ID is already set to a different value");
                }
                fromRepositoryId = repositoryId;
            } else {
                addIf(NOT_BLANK, toRefIds, branch);
                if (toRepositoryId != null && !toRepositoryId.equals(repositoryId)) {
                    throw new IllegalStateException("Repository ID is already set to a different value");
                }
                toRepositoryId = repositoryId;
            }
            return this;
        }

        /**
         * Limits the pull requests to search for by the repository the changes are coming from.
         *
         * @param value the repository id to look for
         * @return the current builder
         */
        @Nonnull
        public Builder fromRepositoryId(@Nullable Integer value) {
            fromRepositoryId = value;

            return this;
        }

        /**
         * Limits the pull requests to search for by the branch the changes are coming from.
         *
         * @param value the branch id
         * @return the current builder
         */
        @Nonnull
        public Builder fromRefId(@Nullable String value) {
            addIf(NOT_BLANK, fromRefIds, value);

            return this;
        }


        /**
         * Limits the pull requests to search for by the branches the changes are coming from.
         *
         * @param values the branch ids
         * @return the current builder
         */
        @Nonnull
        public Builder fromRefIds(@Nullable Iterable<String> values) {
            addIf(NOT_BLANK, fromRefIds, values);

            return this;
        }

        /**
         * The {@link PullRequestOrder order} to return the search results in (defaults to
         * {@link PullRequestOrder#NEWEST NEWEST first}).
         *
         * @param value required order
         * @return the current builder
         */
        @Nonnull
        public Builder order(@Nullable PullRequestOrder value) {
            order = value;

            return this;
        }

        /**
         * Limits the pull requests to search for by the participants they have.
         *
         * @param value the participant
         * @return the current builder
         */
        @Nonnull
        public Builder participant(@Nullable PullRequestParticipantRequest value) {
            addIf(Objects::nonNull, participants, value);

            return this;
        }

        /**
         * Limits the pull requests to search for by the participants they have.
         *
         * @param values the participants
         * @return the current builder
         */
        @Nonnull
        public Builder participants(@Nullable Iterable<PullRequestParticipantRequest> values) {
            addIf(Objects::nonNull, participants, values);

            return this;
        }

        /**
         * Restricts the search to a specified {@link PullRequestState pull request state}.
         *
         * @param value the state to restrict to
         * @return the current builder
         */
        @Nonnull
        public Builder state(@Nullable PullRequestState value) {
            state = value;

            return this;
        }

        /**
         * Limits the pull requests to search for by the repository the changes are going to.
         *
         * @param value the repository id to restrict the search by
         * @return the current builder
         */
        @Nonnull
        public Builder toRepositoryId(@Nullable Integer value) {
            toRepositoryId = value;

            return this;
        }

        /**
         * Limits the pull requests to search for by the branches the changes are going to.
         *
         * @param value the branch id
         * @return the current builder
         */
        @Nonnull
        public Builder toRefId(@Nullable String value) {
            addIf(NOT_BLANK, toRefIds, value);

            return this;
        }

        /**
         * Limits the pull requests to search for by the branches the changes are going to.
         *
         * @param values the branch ids
         * @return the current builder
         */
        @Nonnull
        public Builder toRefIds(@Nullable Iterable<String> values) {
            addIf(NOT_BLANK, toRefIds, values);

            return this;
        }

        /**
         * @param value whether to fetch the {@link PropertyMap properties} for
         *              matching pull requests
         * @return the current builder
         */
        @Nonnull
        public Builder withProperties(boolean value) {
            withProperties = value;

            return this;
        }
    }
}
