package com.atlassian.bitbucket.pull;

import com.atlassian.bitbucket.content.ChangeCallback;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import static java.util.Objects.requireNonNull;

/**
 * Request arguments for {@link PullRequestService#streamChanges(PullRequestChangesRequest, ChangeCallback)}.
 */
public class PullRequestChangesRequest extends AbstractPullRequestRequest {

    private final PullRequestChangeScope changeScope;
    private final String sinceId;
    private final String untilId;
    private final boolean withComments;

    private PullRequestChangesRequest(Builder builder) {
        super(builder);

        changeScope = builder.changeScope;
        sinceId = builder.sinceId;
        untilId = builder.untilId;
        withComments = builder.withComments;
    }

    /**
     * Specifies the scope of the changes to be streamed.
     *
     * @return {@link PullRequestChangeScope#UNREVIEWED UNREVIEWED} to stream unreviewed changes for the
     *         for the current user (if they exist); otherwise, {@link PullRequestChangeScope#ALL ALL} to
     *         stream all changes (the default)
     * @see PullRequestChangeScope
     * @since 4.10
     */
    @Nonnull
    public PullRequestChangeScope getChangeScope() {
        return changeScope;
    }

    /**
     * Specifies the {@code sinceId} to use when retrieving the changes, if the {@link #getChangeScope() scope} is
     * {@link PullRequestChangeScope#RANGE}.
     * <p>
     * If the {@link #getChangeScope() scope} is <strong>not</strong> {@link PullRequestChangeScope#RANGE} then
     * this value is ignored.
     *
     * @return the sinceId to use when retrieving changes
     * @since 5.0
     */
    @Nullable
    public String getSinceId() {
        return sinceId;
    }

    /**
     * Specifies the {@code untilId} to use when retrieving the changes, if the {@link #getChangeScope() scope} is
     * {@link PullRequestChangeScope#RANGE}.
     * <p>
     * If the {@link #getChangeScope() scope} is <strong>not</strong> {@link PullRequestChangeScope#RANGE} then
     * this value is ignored.
     *
     * @return the untilId to use when retrieving changes
     * @since 5.0
     */
    @Nullable
    public String getUntilId() {
        return untilId;
    }

    public boolean isWithComments() {
        return withComments;
    }

    public static class Builder extends AbstractBuilder<Builder> {

        private PullRequestChangeScope changeScope;
        private String sinceId;
        private String untilId;
        private boolean withComments;

        public Builder(@Nonnull PullRequestChangesRequest request) {
            super(requireNonNull(request, "request").getRepositoryId(), request.getPullRequestId());

            changeScope = request.getChangeScope();
            withComments = request.isWithComments();
        }

        public Builder(@Nonnull PullRequest pullRequest) {
            this(requireNonNull(pullRequest, "pullRequest").getToRef().getRepository().getId(), pullRequest.getId());
        }

        public Builder(int repositoryId, long pullRequestId) {
            super(repositoryId, pullRequestId);

            changeScope = PullRequestChangeScope.ALL;
            withComments = true;
        }

        @Nonnull
        public PullRequestChangesRequest build() {
            return new PullRequestChangesRequest(this);
        }

        /**
         * Specifies the scope of the changes to be streamed.
         *
         * @param value {@link PullRequestChangeScope#UNREVIEWED UNREVIEWED} to stream unreviewed changes for the
         *              for the current user (if they exist); otherwise, {@link PullRequestChangeScope#ALL ALL} to
         *              stream all changes (the default)
         * @return {@code this}
         * @see PullRequestChangeScope
         * @since 4.10
         */
        @Nonnull
        public Builder changeScope(@Nonnull PullRequestChangeScope value) {
            changeScope = requireNonNull(value, "changeScope");

            return self();
        }

        /**
         * Specifies the {@code sinceId} to use when retrieving the changes, if the {@link #getChangeScope() scope} is
         * {@link PullRequestChangeScope#RANGE}.
         * <p>
         * If the {@link #getChangeScope() scope} is <strong>not</strong> {@link PullRequestChangeScope#RANGE} then
         * this value is ignored.
         *
         * @param value the sinceId to use when retrieving changes
         * @return {@code this}
         * @since 5.0
         */
        @Nonnull
        public Builder sinceId(@Nullable String value) {
            sinceId = value;

            return self();
        }

        /**
         * Specifies the {@code untilId} to use when retrieving the changes, if the {@link #getChangeScope() scope} is
         * {@link PullRequestChangeScope#RANGE}.
         * <p>
         * If the {@link #getChangeScope() scope} is <strong>not</strong> {@link PullRequestChangeScope#RANGE} then
         * this value is ignored.
         *
         * @param value the untilId to use when retrieving changes
         * @return {@code this}
         * @since 5.0
         */
        @Nonnull
        public Builder untilId(@Nullable String value) {
            untilId = value;

            return self();
        }

        /**
         * Specifies whether comment count attributes should be applied when streaming changes.
         *
         * @param value {@code true} to request comment counts to be applied when streaming changes (the default);
         *              otherwise, {@code false} to omit the counts
         * @return {@code this}
         */
        @Nonnull
        public Builder withComments(boolean value) {
            withComments = value;

            return self();
        }

        @Nonnull
        @Override
        protected Builder self() {
            return this;
        }
    }
}
