package com.atlassian.bitbucket.pull;

import com.atlassian.bitbucket.comment.CommentAction;
import com.google.common.collect.ImmutableSet;
import org.apache.commons.lang3.ArrayUtils;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.Objects;
import java.util.Set;

import static com.google.common.base.Preconditions.checkArgument;

/**
 * Request for searching for pull requests' activities.
 *
 * @see PullRequestSearchRequest
 */
public class PullRequestActivitySearchRequest extends AbstractPullRequestRequest {

    private final Set<CommentAction> commentActions;
    private final Set<Long> commentIds;
    private final Set<PullRequestActivityType> types;
    private final boolean withProperties;

    private PullRequestActivitySearchRequest(Builder builder) {
        super(builder);

        commentActions = builder.commentActions.build();
        commentIds = builder.commentIds.build();
        types = builder.types.build();
        withProperties = builder.withProperties;

        checkArgument(commentActions.isEmpty() || types.contains(PullRequestActivityType.COMMENT),
                "The search request specifies comment actions but does not search for comment activities.");
        checkArgument(commentIds.isEmpty() || types.contains(PullRequestActivityType.COMMENT),
                "The search request specifies comment IDs but does not search for comment activities.");
    }

    /**
     * Limits the <em>comment activities</em> to the ones performing the specified comment actions.
     * <p>
     * Note: this is only valid if the searched {@link #getTypes() types}
     * include {@link PullRequestActivityType#COMMENT comment} activities.
     *
     * @return comment actions the comment activities must perform (can be empty)
     */
    @Nonnull
    public Set<CommentAction> getCommentActions() {
        return commentActions;
    }

    /**
     * Limits the <em>comment activities</em> to the ones affecting the specified comments.
     * <p>
     * Note: this is only valid if the searched {@link #getTypes() types}
     * include {@link PullRequestActivityType#COMMENT comment} activities.
     *
     * @return the IDs of the comments the comment activities affect (can be empty)
     */
    @Nonnull
    public Set<Long> getCommentIds() {
        return commentIds;
    }

    /**
     * Specifies the type(s) of activities to search for.
     *
     * @return the types of activities to retrieve (can be empty)
     * @see PullRequestActivityType
     */
    @Nonnull
    public Set<PullRequestActivityType> getTypes() {
        return types;
    }

    /**
     * @return {@code true} to fetch the pull requests' properties
     */
    public boolean isWithProperties() {
        return withProperties;
    }

    public static class Builder extends AbstractBuilder<Builder> {

        private final ImmutableSet.Builder<CommentAction> commentActions = ImmutableSet.builder();
        private final ImmutableSet.Builder<Long> commentIds = ImmutableSet.builder();
        private final ImmutableSet.Builder<PullRequestActivityType> types = ImmutableSet.builder();

        private boolean withProperties;

        public Builder(@Nonnull PullRequest pullRequest) {
            super(pullRequest);
        }

        public Builder(int repositoryId, long pullRequestId) {
            super(repositoryId, pullRequestId);
        }

        @Nonnull
        public PullRequestActivitySearchRequest build() {
            return new PullRequestActivitySearchRequest(this);
        }

        @Nonnull
        public Builder commentActions(@Nullable CommentAction value, @Nullable CommentAction... values) {
            addIf(Objects::nonNull, commentActions, value, values);

            return self();
        }

        @Nonnull
        public Builder commentActions(@Nullable Iterable<CommentAction> values) {
            addIf(Objects::nonNull, commentActions, values);

            return self();
        }

        /**
         * @param value  the first comment ID to search for activities for
         * @param values additional comment IDs to search for activities for
         * @return {@code this}
         */
        @Nonnull
        public Builder commentIds(long value, long... values) {
            commentIds.add(value);
            if (!(values == null || values.length == 0)) {
                commentIds.add(ArrayUtils.toObject(values));
            }

            return self();
        }

        @Nonnull
        public Builder commentIds(@Nullable Iterable<Long> values) {
            addIf(Objects::nonNull, commentIds, values);

            return self();
        }

        @Nonnull
        public Builder types(@Nullable PullRequestActivityType value, @Nullable PullRequestActivityType... values) {
            addIf(Objects::nonNull, types, value, values);

            return self();
        }

        @Nonnull
        public Builder types(@Nullable Iterable<PullRequestActivityType> values) {
            addIf(Objects::nonNull, types, values);

            return self();
        }

        /**
         * @param value {@code true} to load {@link PullRequest#getProperties properties} for the pull requests
         *              on each matched activity; otherwise, {@code false} to omit them
         * @return {@code this}
         */
        @Nonnull
        public Builder withProperties(boolean value) {
            withProperties = value;

            return self();
        }

        @Nonnull
        @Override
        protected Builder self() {
            return this;
        }
    }
}
