package com.atlassian.bitbucket.markup;

import com.google.common.collect.ImmutableMap;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.HashMap;
import java.util.Map;

import static java.util.Objects.requireNonNull;

/**
 * Context-specific data for rendering markup.
 * <p>
 * This information will be used by rendering components to determine context-specific information,
 * such as whether URLs are absolute or relative.
 */
public class RenderContext {

    private final Boolean includeHeadingId;
    private final Boolean hardwrap;
    private final Boolean htmlEscaped;
    private final UrlMode urlMode;
    private final Map<String, Object> data;

    private RenderContext(Builder builder) {
        this.hardwrap = builder.hardwrap;
        this.htmlEscaped = builder.htmlEscaped;
        this.includeHeadingId = builder.includeHeadingId;
        this.urlMode = builder.urlMode;
        this.data = ImmutableMap.copyOf(builder.data);
    }

    /**
     * @return additional data to make available when rendering, which may be empty but never {@code null}
     */
    @Nonnull
    public Map<String, Object> getData() {
        return data;
    }

    /**
     * @return {@code TRUE} to render newlines as breaks; {@code FALSE} to retain newlines unchanged; or {@code null}
     *         to use the system-configured default (currently {@code true})
     */
    @Nullable
    public Boolean getHardwrap() {
        return hardwrap;
    }

    /**
     * @return {@code TRUE} to escape HTML in the content when rendering; {@code FALSE} to retain HTML unchanged; or
     *         {@code null} to use the system-configured default (currently {@code true})
     */
    @Nullable
    public Boolean getHtmlEscaped() {
        return htmlEscaped;
    }

    /**
     * @return {@code TRUE} to include an ID on each rendered heading; {@code FALSE} to omit them; or {@code null}
     *         to use the system-configured default (currently {@code false})
     * @since 5.11
     */
    @Nullable
    public Boolean getIncludeHeadingId() {
        return includeHeadingId;
    }

    /**
     * @return the mode to use when rendering URLs ({@code RELATIVE} by default)
     */
    @Nonnull
    public UrlMode getUrlMode() {
        return urlMode;
    }

    public static class Builder {

        private final Map<String, Object> data;

        private Boolean hardwrap;
        private Boolean htmlEscaped;
        private Boolean includeHeadingId;
        private UrlMode urlMode;

        public Builder() {
            urlMode = UrlMode.RELATIVE;
            data = new HashMap<>();
        }

        public Builder(@Nonnull RenderContext context) {
            requireNonNull(context, "context");

            data = new HashMap<>(context.getData());
            hardwrap = context.getHardwrap();
            htmlEscaped = context.getHtmlEscaped();
            includeHeadingId = context.getIncludeHeadingId();
            urlMode = context.getUrlMode();
        }

        /**
         * @return the built context
         */
        @Nonnull
        public RenderContext build() {
            return new RenderContext(this);
        }

        /**
         * Clears any provided {@link #data(Map) data}.
         *
         * @return {@code this}
         */
        @Nonnull
        public Builder clearData() {
            data.clear();

            return this;
        }

        /**
         * @param value a map containing additional data to make available when rendering, which is <i>appended</i>
         *              to any existing data
         * @return {@code this}
         */
        @Nonnull
        public Builder data(@Nonnull Map<String, Object> value) {
            data.putAll(requireNonNull(value, "data"));

            return this;
        }

        /**
         * @param value {@code true} to convert newlines to breaks; otherwise, {@code false} to leave them intact
         * @return {@code this}
         */
        @Nonnull
        public Builder hardwrap(boolean value) {
            hardwrap = value;

            return this;
        }

        /**
         * @param value {@code TRUE} to convert newlines to breaks; {@code FALSE} to leave them intact; or
         *              {@code null} to use the system-configured default
         * @return {@code this}
         * @since 5.11
         */
        @Nonnull
        public Builder hardwrap(@Nullable Boolean value) {
            hardwrap = value;

            return this;
        }

        /**
         * @param value {@code true} if HTML should be escaped in the input markup; otherwise, {@code false} to
         *              retain HTML
         * @return {@code this}
         */
        @Nonnull
        public Builder htmlEscape(boolean value) {
            htmlEscaped = value;

            return this;
        }

        /**
         * @param value {@code TRUE} if HTML should be escaped in the input markup; {@code FALSE} to retain HTML;
         *              or {@code null} to use the system-configured default
         * @return {@code this}
         * @since 5.11
         */
        @Nonnull
        public Builder htmlEscape(Boolean value) {
            htmlEscaped = value;

            return this;
        }

        /**
         * @param value {@code true} if an ID should be added for each rendered header rendered; otherwise,
         *              {@code false} to render headers without IDs
         * @return {@code this}
         * @since 5.11
         */
        @Nonnull
        public Builder includeHeadingId(boolean value) {
            includeHeadingId = value;

            return this;
        }

        /**
         * @param value {@code TRUE} if an ID should be added for each rendered header rendered; {@code FALSE} to
         *              render headers without IDs; or {@code null} to use the system-configured default
         * @return {@code this}
         * @since 5.11
         */
        @Nonnull
        public Builder includeHeadingId(Boolean value) {
            includeHeadingId = value;

            return this;
        }

        /**
         * @param value the mode to use when rendering URLs
         * @return {@code this}
         */
        @Nonnull
        public Builder urlMode(@Nonnull UrlMode value) {
            urlMode = requireNonNull(value, "urlMode");

            return this;
        }
    }
}
