package com.atlassian.bitbucket.license;

import com.atlassian.bitbucket.i18n.KeyedMessage;
import com.atlassian.bitbucket.permission.Permission;
import com.atlassian.bitbucket.user.ApplicationUser;
import com.atlassian.extras.api.bitbucket.BitbucketServerLicense;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.security.Principal;

/**
 * Provides license related functions.
 */
public interface LicenseService {

    /**
     * Check whether the license is valid, and operations are permitted.
     *
     * @return {@code true} if the license is valid and not exceeded, or in a grace period, {@code false} otherwise
     * @since 4.8
     */
    boolean isLicenseValid();

    /**
     * Check whether the current instance has a license.
     *
     * @return {@code true} if the instance has an license (<em>note: this will also return
     * {@code true} if the license has expired.</em>)
     */
    boolean isPresent();

    /**
     * Get the license of this instance.
     *
     * @return license of this instance
     */
    @Nullable
    BitbucketServerLicense get();

    /**
     * Get the encoded license for this instance. If no license has been set, this will return {@code null}.
     *
     * @return the encoded license string, or {@code null} if no license is set
     */
    @Nullable
    String getAsString();

    /**
     * Get the number of licensed users currently in the system.
     *
     * @return The count of licensed users.
     */
    int getLicensedUsersCount();

    /**
     * Add or replace the license of this instance.
     *
     * @param license new license of this instance
     * @return the new license
     */
    @Nonnull
    BitbucketServerLicense set(@Nonnull String license);

    /**
     * Get a message regarding the validity of the currently configured license
     *
     * @return null if the license is valid, otherwise a descriptive message about why the license is invalid
     */
    KeyedMessage getValidityMessage();

    /**
     * Get a message if the license is over its limits
     *
     * @return null if the system is within the license limits, otherwise a descriptive message about how the license
     *              limits have been exceeeded
     */
    KeyedMessage getOverLimitMessage();

    /**
     * Checks both the license validity and its limits.
     *
     * @return a message if the license is invalid or is over its limits.
     */
    KeyedMessage getStatus();

    /**
     * Check if a user is active and licensed to use the application.
     *
     * @param principal identifier for the user
     * @return {@code true} if the user is both active and {@link Permission#LICENSED_USER licensed}
     */
    boolean canLogin(Principal principal);

    /**
     * Validate if the given user can be licensed without violating license limits
     *
     * @param user       the user to be licensed
     * @param permission the permission to be granted to the user
     * @throws LicenseLimitException if granting the permission to the specified user would exceed the number of
     *                               licensed users allowed by the current license
     */
    void validateCanLicenseUser(ApplicationUser user, Permission permission) throws LicenseLimitException;

    /**
     * Validate if the specified group can be licensed without violating license limits. Licensing a group licenses all
     * of the users the group contains.
     *
     * @param group      the group to be licensed
     * @param permission the permission to be granted to the group
     * @throws LicenseLimitException if granting the permission to the specified group, and all the users is contains,
     *                               would exceed the number of licensed users allowed by the current license
     */
    void validateCanLicenseGroup(String group, Permission permission) throws LicenseLimitException;

    /**
     * Validate if the named user can be added to the specified group without violating license limits.
     * <p>
     * If the group is not does not grant {@link Permission#LICENSED_USER LICENSED_USER}, this check will always
     * pass. Otherwise, it validates that licensing an additional user would comply with license limits and will
     * throw a {@link LicenseLimitException} if not.
     *
     * @param username the user to be added to the group
     * @param group    the group the user is to be added to
     * @throws LicenseLimitException if adding the user to the group would exceed the number of licensed users allowed
     *                               by the current license
     */
    void validateCanAddUserToGroup(String username, String group) throws LicenseLimitException;
}
