package com.atlassian.bitbucket.job;

import com.atlassian.bitbucket.user.ApplicationUser;
import com.google.common.base.MoreObjects;
import com.google.common.base.Objects;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.Optional;

import static java.util.Objects.requireNonNull;
import static java.util.Optional.ofNullable;
import static org.apache.commons.lang3.StringUtils.trimToNull;

/**
 * A request to create a {@link Job job}
 *
 * @since 5.13
 */
public class JobCreationRequest {

    private final ApplicationUser initiator;
    private final int progressPercentage;
    private final String progressMessage;
    private final JobState state;
    private final String type;

    private JobCreationRequest(Builder builder) {
        initiator = builder.initiator;
        progressPercentage = builder.progressPercentage;
        progressMessage = builder.progressMessage;
        state = builder.state;
        type = requireNonNull(builder.type, "type");
    }

    @Nonnull
    public Optional<ApplicationUser> getInitiator() {
        return ofNullable(initiator);
    }

    @Nonnull
    public Optional<String> getProgressMessage() {
        return ofNullable(progressMessage);
    }

    public int getProgressPercentage() {
        return progressPercentage;
    }

    @Nonnull
    public JobState getState() {
        return state;
    }

    @Nonnull
    public String getType() {
        return type;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        JobCreationRequest that = (JobCreationRequest) o;
        return getProgressPercentage() == that.getProgressPercentage() &&
                Objects.equal(getInitiator().map(ApplicationUser::getId), that.getInitiator().map(ApplicationUser::getId)) &&
                Objects.equal(getProgressMessage(), that.getProgressMessage()) &&
                Objects.equal(getState(), that.getState()) &&
                Objects.equal(getType(), that.getType());
    }

    @Override
    public int hashCode() {
        return Objects.hashCode(getInitiator().map(ApplicationUser::getId), getProgressPercentage(), getProgressMessage(),
                getState(), getType());
    }

    @Override
    public String toString() {
        return MoreObjects.toStringHelper(this)
                .add("initiator", initiator)
                .add("progressPercentage", progressPercentage)
                .add("progressMessage", progressMessage)
                .add("state", state)
                .add("type", type)
                .toString();
    }

    public static class Builder {

        private ApplicationUser initiator;
        private JobState state = JobState.INITIALISING;
        private String type;
        private int progressPercentage;
        private String progressMessage;

        @Nonnull
        public JobCreationRequest build() {
            return new JobCreationRequest(this);
        }

        @Nonnull
        public Builder initiator(@Nullable ApplicationUser value) {
            initiator = value;
            return this;
        }

        @Nonnull
        public Builder progressMessage(@Nullable String value) {
            progressMessage = trimToNull(value);
            return this;
        }

        @Nonnull
        public Builder progressPercentage(int value) {
            progressPercentage = Math.max(0, Math.min(value, 100));
            return this;
        }

        @Nonnull
        public Builder state(@Nonnull JobState value) {
            state = requireNonNull(value, "state");
            return this;
        }

        @Nonnull
        public Builder type(@Nonnull String value) {
            type = requireNonNull(trimToNull(value), "type");
            return this;
        }
    }
}