package com.atlassian.bitbucket.event.pull;

import com.atlassian.bitbucket.pull.PullRequest;
import com.atlassian.bitbucket.pull.PullRequestAction;
import com.atlassian.bitbucket.pull.PullRequestService;
import com.atlassian.bitbucket.user.ApplicationUser;
import com.atlassian.event.api.AsynchronousPreferred;
import com.google.common.collect.ImmutableSet;

import javax.annotation.Nonnull;
import java.util.Set;

/**
 * An event raised when the reviewers of a pull request are updated.
 * <p>
 * {@link #getAddedReviewers()} and {@link #getRemovedReviewers()} allow tracking who has been
 * {@link PullRequestService#addReviewer(int, long, String) added} and
 * {@link PullRequestService#removeReviewer(int, long, String) removed}, respectively.
 * @since 4.4
 */
@AsynchronousPreferred
public class PullRequestReviewersUpdatedEvent extends PullRequestEvent {

    private final Set<ApplicationUser> addedReviewers;
    private final Set<ApplicationUser> removedReviewers;

    public PullRequestReviewersUpdatedEvent(@Nonnull Object source, @Nonnull PullRequest pullRequest,
                                            @Nonnull Set<ApplicationUser> added, @Nonnull Set<ApplicationUser> removed) {
        super(source, pullRequest, PullRequestAction.UPDATED);

        addedReviewers = ImmutableSet.copyOf(added);
        removedReviewers = ImmutableSet.copyOf(removed);
    }

    /**
     * @return the set of new or existing participants assigned to the reviewer role
     */
    @Nonnull
    public Set<ApplicationUser> getAddedReviewers() {
        return addedReviewers;
    }

    /**
     * @return the set of participants unassigned from the reviewer role
     */
    @Nonnull
    public Set<ApplicationUser> getRemovedReviewers() {
        return removedReviewers;
    }
}
