package com.atlassian.bitbucket.event.branch;

import com.atlassian.bitbucket.hook.repository.RepositoryHookRequest;
import com.atlassian.bitbucket.hook.repository.StandardRepositoryHookTrigger;
import com.atlassian.bitbucket.repository.*;
import org.apache.commons.lang3.StringUtils;

import javax.annotation.Nonnull;
import java.util.Collection;
import java.util.Collections;

/**
 * A {@link RepositoryHookRequest hook request} that is raised just before a branch is deleted using the UI or REST.
 * Branches deleted by pushing will trigger a standard {@link RepositoryHookRequest} and not this specific type.
 *
 * @see StandardRepositoryHookTrigger#BRANCH_DELETE
 * @since 5.0
 */
public class BranchDeletionHookRequest extends BranchHookRequest {

    private final Collection<RefChange> changes;

    private BranchDeletionHookRequest(Builder builder) {
        super(builder);

        changes = Collections.singleton(new SimpleRefChange.Builder()
                .from(getBranch())
                .toHash(StringUtils.repeat("0", 40))
                .type(RefChangeType.DELETE)
                .build());
    }

    @Nonnull
    @Override
    public Collection<RefChange> getRefChanges() {
        return changes;
    }

    public static class Builder extends AbstractBuilder<Builder> {

        public Builder(@Nonnull Repository repository, @Nonnull Branch branch) {
            super(repository, StandardRepositoryHookTrigger.BRANCH_CREATE, branch);
        }

        @Nonnull
        public BranchDeletionHookRequest build() {
            return new BranchDeletionHookRequest(this);
        }

        @Nonnull
        @Override
        protected Builder self() {
            return this;
        }
    }
}
