package com.atlassian.bitbucket.content;

import com.atlassian.bitbucket.io.InputSupplier;
import com.atlassian.bitbucket.repository.Repository;
import com.atlassian.bitbucket.util.BuilderSupport;
import org.apache.commons.lang3.StringUtils;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.io.InputStream;
import java.util.Optional;

import static java.util.Objects.requireNonNull;
import static java.util.Optional.ofNullable;

/**
 * Represents a request to edit a file, this request can be used to create a new file as well as updating an
 * existing file.
 * <p>
 * The following properties are <i>required</i>:
 * <ul>
 * <li>{@link #getBranch()} ()}: The name of the branch on which the edit should happen</li>
 * <li>{@link #getPath()} ()}: The path of the file</li>
 * <li>{@link #getRepository()}: The repository in which the edit operation should be done</li>
 * <li>{@link #getContent()}: A supplier providing access to the file content</li>
 * </ul>
 * If {@link #getSourceCommitId()} is not provided, it's assumed the file doesn't exist on the branch and a
 * new file will be created.
 *
 * @see ContentService#editFile(EditFileRequest)
 * @since 4.13
 */
public class EditFileRequest {

    private final String branch;
    private final InputSupplier<InputStream> content;
    private final String message;
    private final String path;
    private final Repository repository;
    private final String sourceBranch;
    private final String sourceCommitId;

    private EditFileRequest(Builder builder) {
        branch = builder.branch;
        content = builder.content;
        message = builder.message;
        path = builder.path;
        repository = builder.repository;
        sourceCommitId = builder.sourceCommitId;
        sourceBranch = StringUtils.defaultString(builder.sourceBranch, builder.branch);
    }

    /**
     * @return the branch on which the edit operation would happen
     */
    @Nonnull
    public String getBranch() {
        return branch;
    }

    /**
     * @return a supplier providing access to the file content
     */
    @Nonnull
    public InputSupplier<InputStream> getContent() {
        return content;
    }

    /**
     * @return the commit message
     */
    @Nonnull
    public Optional<String> getMessage() {
        return ofNullable(message);
    }

    /**
     * @return the file path
     */
    @Nonnull
    public String getPath() {
        return path;
    }

    /**
     * @return the repository in which the edit operation would be executed
     */
    @Nonnull
    public Repository getRepository() {
        return repository;
    }

    /**
     * @return the name of the branch that should serve as the start point for the edit operation
     *
     * @since 5.2
     */
    @Nonnull
    public String getSourceBranch() {
        return sourceBranch;
    }

    /**
     * @return the commit ID of the existing file in the repository, or {@code Optional#empty} if this is a new file
     */
    @Nonnull
    public Optional<String> getSourceCommitId() {
        return ofNullable(sourceCommitId);
    }

    public static class Builder extends BuilderSupport {

        private final String branch;
        private final String path;
        private final Repository repository;
        private InputSupplier<InputStream> content;
        private String message;
        private String sourceBranch;
        private String sourceCommitId;

        public Builder(@Nonnull String branch, @Nonnull String path, @Nonnull Repository repository) {
            this.branch = requireNonBlank(branch, "branch");
            this.path = requireNonBlank(path, "path");
            this.repository = requireNonNull(repository, "repository");
        }

        @Nonnull
        public EditFileRequest build() {
            requireNonNull(content, "content");
            return new EditFileRequest(this);
        }

        @Nonnull
        public Builder content(@Nonnull InputSupplier<InputStream> value) {
            content = requireNonNull(value, "content");
            return this;
        }

        @Nonnull
        public Builder message(@Nullable String value) {
            message = value;
            return this;
        }

        /**
         * @param value the name of the branch that should serve as the starting point for the edit operation. If not
         *              set, this defaults to the branch provided in the constructor
         * @return the builder
         *
         * @since 5.2
         */
        @Nonnull
        public Builder sourceBranch(@Nullable String value) {
            sourceBranch = value;
            return this;
        }

        /**
         * @param value the commit ID that the file was edited on, or {@code null} if this is a new file.
         * @return the builder
         */
        @Nonnull
        public Builder sourceCommitId(@Nullable String value) {
            sourceCommitId = value;
            return this;
        }
    }
}
