package com.atlassian.bitbucket.content;

import javax.annotation.Nonnull;
import java.util.List;
import java.util.Optional;

import static java.util.Optional.empty;

/**
 * Describes a single hunk, with one or more segments, in a diff.
 */
public interface DiffHunk {

    /**
     * Retrieves the context snippet for the hunk, which may be {@code empty()} depending on the SCM's ability to
     * detect it.
     * <p>
     * Context is chosen by the SCM, and is based on the SCM's understanding of the syntax of the file. Depending
     * on the syntax used, the SCM may not be able to detect the context or it may detect "incorrect" context.
     *
     * @return a context snippet for the hunk, which may be {@code empty()} if the SCM could not detect any
     * @since 5.5
     */
    @Nonnull
    default Optional<String> getContext() {
        return empty();
    }

    /**
     * @return the starting line for this hunk in the destination file
     */
    int getDestinationLine();

    /**
     * @return the number of lines this hunk spans in the destination file
     */
    int getDestinationSpan();

    /**
     * @return the segments included in this hunk
     */
    @Nonnull
    List<DiffSegment> getSegments();

    /**
     * @return the starting line for this hunk in the source file
     */
    int getSourceLine();

    /**
     * @return the number of lines this hunk spans in the source file
     */
    int getSourceSpan();

    /**
     * @return {@code true} if one or more <i>entire segments</i> were omitted due to the size of the hunk, or
     *         if any lines were omitted from an included segment; otherwise, {@code false} if all of the segments
     *         and lines in the hunk are included
     */
    boolean isTruncated();
}
