package com.atlassian.bitbucket.content;

import com.atlassian.bitbucket.scm.CommandResult;
import com.atlassian.bitbucket.util.AbstractSummary;

import javax.annotation.Nonnull;

/**
 * Summarizes {@link Change changes} which have been streamed to a {@link ChangeCallback}.
 */
public class ChangeSummary extends AbstractSummary {

    private final boolean truncated;

    private ChangeSummary(Builder builder) {
        super(builder);

        this.truncated = builder.truncated;
    }

    /**
     * @return {@code true} if one or more {@link Change changes} were not streamed due to <i>internal hard limits</i>
     *         on the number of changes callbacks can receive; otherwise, {@code false} if all changes were streamed
     */
    public boolean isTruncated() {
        return truncated;
    }

    public static class Builder extends AbstractBuilder<Builder, ChangeSummary> {

        private boolean truncated;

        /**
         * Creates and initializes a new builder using details from the provided {@link AbstractSummary summary}. If
         * the provided summary is a {@link ChangeSummary}, those details will also be copied over.
         *
         * @param summary the summary to initialize this builder from
         */
        public Builder(@Nonnull AbstractSummary summary) {
            super(summary);

            if (summary instanceof ChangeSummary) {
                truncated = ((ChangeSummary) summary).isTruncated();
            }
        }

        /**
         * Creates a new builder with the provided {@link CommandResult result}.
         *
         * @param result the operation result for the summary
         */
        public Builder(@Nonnull CommandResult result) {
            super(result);
        }

        @Nonnull
        @Override
        public ChangeSummary build() {
            return new ChangeSummary(this);
        }

        @Nonnull
        public Builder truncated(boolean value) {
            truncated = value;

            return self();
        }

        @Nonnull
        @Override
        protected Builder self() {
            return this;
        }
    }
}
