package com.atlassian.bitbucket.content;

import com.atlassian.bitbucket.scm.ScmExtendedCommandFactory;
import com.atlassian.bitbucket.scm.ScmFeature;
import com.google.common.collect.ImmutableSet;

import javax.annotation.Nonnull;
import java.util.Locale;
import java.util.Set;
import java.util.stream.Stream;

import static java.util.Objects.requireNonNull;

/**
 * Describes the different formats {@link ScmExtendedCommandFactory#archive archives} may be streamed in.
 * <p>
 * Individual SCMs may support additional formats, but any SCM which advertizes the {@link ScmFeature#ARCHIVE archive
 * feature} is <i>required</i> to support all of these formats. Using any other format requires referencing the SCM
 * directly.
 *
 * @since 5.1
 */
public enum ArchiveFormat {

    /**
     * Streams the archive as an <i>uncompressed</i> tarball.
     */
    TAR("tar"),
    /**
     * Streams the archive as a GZip-compressed tarball.
     */
    TAR_GZ("tar.gz", "tgz"),
    /**
     * Streams the archive as a zip file, using the default compression level for the SCM.
     */
    ZIP("zip");

    private final String defaultExtension;
    private final Set<String> extensions;

    ArchiveFormat(String... extensions) {
        this.extensions = ImmutableSet.copyOf(extensions);

        defaultExtension = extensions[0];
    }

    /**
     * Searches for the format which produces the provided {@code extension}, <i>case-insensitively</i>.
     *
     * @param extension the extension to search for
     * @return the format which produces the provided {@code extension}
     * @throws IllegalArgumentException if no format produces the requested extension
     */
    @Nonnull
    public static ArchiveFormat fromExtension(@Nonnull String extension) {
        String filterExtension = requireNonNull(extension, "extension").toLowerCase(Locale.ROOT);

        return Stream.of(values())
                .filter(format -> format.getExtensions().contains(filterExtension))
                .findFirst()
                .orElseThrow(() -> new IllegalArgumentException("No ArchiveFormat is associated with extension [" + extension + "]"));
    }

    /**
     * @return the default extension for this format
     */
    @Nonnull
    public String getDefaultExtension() {
        return defaultExtension;
    }

    /**
     * @return a set containing one or more extensions which are commonly used with this format
     */
    @Nonnull
    public Set<String> getExtensions() {
        return extensions;
    }
}
