package com.atlassian.bitbucket.content;

import com.atlassian.bitbucket.util.BuilderSupport;
import com.google.common.collect.ImmutableSet;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.Set;

import static com.google.common.base.Preconditions.checkNotNull;

public abstract class AbstractDiffRequest {

    /**
     * Indicates the request should supply the default number of context lines for added and removed lines. The
     * default is configurable, so it may be changed by a system administrator. The shipped setting is 10 lines.
     */
    public static final int DEFAULT_CONTEXT_LINES = -1;

    private final int contextLines;
    private final DiffContentFilter filter;
    private final Set<String> paths;
    private final DiffWhitespace whitespace;

    protected AbstractDiffRequest(AbstractBuilder<?, ?> builder) {
        contextLines = builder.contextLines;
        filter = builder.filter;
        paths = builder.paths.build();
        whitespace = builder.whitespace;
    }

    /**
     * Retrieves the numbers of context lines to show around added and removed lines. If the value is {@code 0} context
     * will be omitted and only added and removed lines will be shown. For positive values, that number of lines will
     * be shown. For negative values, the <i>system-configured default</i> number will be shown.
     *
     * @return the number of context lines that should be included around added and removed lines in the diff
     */
    // the method was originally added in 2.11 (on DiffRequest), before the creation of this superclass
    public int getContextLines() {
        return contextLines;
    }

    /**
     * @return an optional filter for this diff to filter out lines (and their surrounding file/hunk/segment)
     */
    @Nullable
    public DiffContentFilter getFilter() {
        return filter;
    }

    /**
     * @return specific paths to diff, which may be empty to stream the diff for every modified path
     */
    @Nonnull
    public Set<String> getPaths() {
        return paths;
    }

    /**
     * @return the whitespace settings used in the diff
     */
    @Nonnull
    public DiffWhitespace getWhitespace() {
        return whitespace;
    }

    /**
     * Retrieves a flag indicating whether an explicit number of context lines has been requested. When this method
     * returns {@code true}, {@link #getContextLines()} will return a value greater than or equal to {@code 0}, When
     * it returns {@code false}, {@link #getContextLines()} will return {@link #DEFAULT_CONTEXT_LINES}.
     *
     * @return {@code true} if a non-{@link #DEFAULT_CONTEXT_LINES default} number of context lines has been requested;
     *         otherwise, {@code false} to use the default number of lines
     */
    public boolean hasContextLines() {
        return contextLines > DEFAULT_CONTEXT_LINES;
    }

    protected abstract static class AbstractBuilder<B extends AbstractBuilder<B, R>, R> extends BuilderSupport {

        private final ImmutableSet.Builder<String> paths = ImmutableSet.builder();

        private int contextLines;
        private DiffContentFilter filter;
        private DiffWhitespace whitespace;

        protected AbstractBuilder() {
            this.contextLines = DEFAULT_CONTEXT_LINES;
            this.whitespace = DiffWhitespace.SHOW;
        }

        protected AbstractBuilder(@Nonnull AbstractDiffRequest request) {
            this.contextLines = checkNotNull(request, "request").getContextLines();
            this.whitespace = checkNotNull(request.getWhitespace(), "request.whitespace");

            addIf(NOT_BLANK, paths, request.getPaths());
        }

        @Nonnull
        public abstract R build();

        /**
         * Specifies the number of context lines to include around added/removed lines. {@code 0} and positive values
         * are treated as the number of lines to request. <i>Any negative value</i> is treated as a request for the
         * <i>system-configured default</i> number of lines.
         * <p>
         * When requesting the default number of context lines, it is encouraged to use {@link #DEFAULT_CONTEXT_LINES}
         * for clarity: {@code contextLines(DiffRequest.DEFAULT_CONTEXT_LINES)}.
         *
         * @param value the number of context lines to include around added and removed lines in the diff, which may
         *              be {@link #DEFAULT_CONTEXT_LINES} to use the default number
         * @return {@code this}
         */
        @Nonnull
        public B contextLines(int value) {
            contextLines = Math.max(DEFAULT_CONTEXT_LINES, value);

            return self();
        }

        @Nonnull
        public B filter(@Nullable DiffContentFilter value) {
            filter = value;
            return self();
        }

        @Nonnull
        public B path(@Nullable String value) {
            addIf(NOT_BLANK, paths, value);

            return self();
        }

        @Nonnull
        public B paths(@Nullable Iterable<String> values) {
            addIf(NOT_BLANK, paths, values);

            return self();
        }

        @Nonnull
        public B paths(@Nullable String value, @Nullable String... values) {
            addIf(NOT_BLANK, paths, value, values);

            return self();
        }

        @Nonnull
        public B whitespace(@Nonnull DiffWhitespace value) {
            whitespace = checkNotNull(value, "value");

            return self();
        }

        @Nonnull
        protected abstract B self();
    }
}
