package com.atlassian.bitbucket.commit;

import com.atlassian.bitbucket.scm.CommandResult;
import com.atlassian.bitbucket.util.AbstractSummary;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.Optional;

import static java.util.Optional.ofNullable;

/**
 * Summarizes the {@link LastModifiedRequest request}, providing the {@link #getLatestCommit latest commit} to
 * modify the {@link LastModifiedRequest#getPath requested path}. This commit is handled separately because it
 * may be different from any of the commits that last modified any of the files in the path.
 *
 * @since 4.6
 */
public class LastModifiedSummary extends AbstractSummary {

    private final Commit newestCommit;

    private LastModifiedSummary(Builder builder) {
        super(builder);

        newestCommit = builder.newestCommit;
    }

    /**
     * @return the last commit to modify the {@link LastModifiedRequest#getPath requested path}, which may be
     *         {@code empty()}, if the {@link #getResult command failed}, but will never be {@code null}
     */
    @Nonnull
    public Optional<Commit> getLatestCommit() {
        return ofNullable(newestCommit);
    }

    public static class Builder extends AbstractBuilder<Builder, LastModifiedSummary> {

        private Commit newestCommit;

        public Builder(@Nonnull CommandResult result) {
            super(result);
        }

        @Nonnull
        @Override
        public LastModifiedSummary build() {
            return new LastModifiedSummary(this);
        }

        @Nonnull
        public Builder latestCommit(@Nullable Commit value) {
            newestCommit = value;

            return self();
        }

        @Nonnull
        @Override
        protected Builder self() {
            return this;
        }
    }
}
