package com.atlassian.bitbucket.comment;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import static java.util.Objects.requireNonNull;

/**
 * Request to create or update a {@link CommentThreadDiffAnchor#isFileAnchor() file comment}.
 *
 * @since 5.0
 */
public class AddFileCommentRequest extends AddCommentRequest {

    private final CommentThreadDiffAnchorType diffType;
    private final String fromHash;
    private final String path;
    private final String srcPath;
    private final String toHash;

    protected AddFileCommentRequest(@Nonnull AbstractFileBuilder builder) {
        super(builder);

        diffType = builder.diffType;
        fromHash = builder.fromHash;
        path = builder.path;
        srcPath = builder.srcPath;
        toHash = builder.toHash;
    }

    /**
     * @return the type of diff the comment is being added on
     */
    @Nonnull
    public CommentThreadDiffAnchorType getDiffType() {
        return diffType;
    }

    /**
     * @return the fromHash for the diff onto which the comment is to be anchored. A {@code null} value will prompt
     *         the system to try and resolve the most suitable fromHash which can be {@code null} for comments made on
     *         a {@link com.atlassian.bitbucket.commit.CommitDiscussion commit} which doesn't have a parent
     *         (i.e.: initial commits).
     */
    @Nullable
    public String getFromHash() {
        return fromHash;
    }

    /**
     * @return the path to the file on which the comment is to be added
     */
    @Nonnull
    public String getPath() {
        return path;
    }

    /**
     * @return the path the file on which the comment is to be added used to have
     */
    @Nullable
    public String getSrcPath() {
        return srcPath;
    }

    /**
     * @return the toHash for the diff onto which the comment is to be anchored.
     */
    @Nullable
    public String getToHash() {
        return toHash;
    }

    public static class Builder extends AbstractFileBuilder<Builder, AddFileCommentRequest> {

        public Builder(@Nonnull Commentable commentable, @Nonnull String text,
                       @Nonnull CommentThreadDiffAnchorType diffType, @Nonnull String path) {
            super(commentable, text, diffType, path);
        }

        @Nonnull
        @Override
        public AddFileCommentRequest build() {
            return new AddFileCommentRequest(this);
        }

        @Nonnull
        @Override
        protected Builder self() {
            return this;
        }
    }

    protected abstract static class AbstractFileBuilder<B extends AbstractFileBuilder<B, R>, R extends AddFileCommentRequest>
            extends AbstractBuilder<B, R> {

        private final CommentThreadDiffAnchorType diffType;

        private String fromHash;
        private String path;
        private String srcPath;
        private String toHash;

        public AbstractFileBuilder(@Nonnull Commentable commentable, @Nonnull String text,
                                   @Nonnull CommentThreadDiffAnchorType diffType, @Nonnull String path) {
            super(commentable, text);

            this.diffType = requireNonNull(diffType, "diffType");
            this.path = requireNonBlank(path, "path");
        }

        @Nonnull
        public B fromHash(@Nullable String value) {
            fromHash = value;

            return self();
        }

        @Nonnull
        public B srcPath(@Nullable String value) {
            srcPath = value;

            return self();
        }

        @Nonnull
        public B toHash(@Nullable String value) {
            toHash = value;

            return self();
        }
    }
}
