package com.atlassian.bitbucket.avatar;

import javax.validation.constraints.Min;
import java.util.Objects;

import static com.google.common.base.Preconditions.checkArgument;

public class AvatarRequest {

    private final boolean secure;
    private final int size;
    private final boolean useConfigured;

    /**
     * {@code useConfigured} is defaulted to {@code false}
     * @see #AvatarRequest(boolean, int, boolean)
     */
    public AvatarRequest(boolean secure, @Min(1) int size) {
        this(secure, size, false);
    }

    /**
     * Constructs a new {@code AvatarRequest} for the provided {@code scheme} and {@code size}.
     * <p>
     * The {@code secure} flag allows the caller to control whether avatar URLs use HTTP or HTTPS. The {@code size}
     * controls the dimensions of the avatar returned.
     *
     * @param secure {@code true} if avatar URLs should use HTTPS; otherwise, {@code false} for HTTP
     * @param size   the height/width of the avatar being requested
     * @param useConfigured whether to use the configured base URL; if {@code false} the request context will be used instead
     * @throws IllegalArgumentException if the provided {@code size} is less than 1.
     * @throws NullPointerException if the provided {@code scheme} is {@code null}.
     */
    public AvatarRequest(boolean secure, @Min(1) int size, boolean useConfigured) {
        checkArgument(size > 0);

        this.secure = secure;
        this.size = size;
        this.useConfigured = useConfigured;
    }

    /**
     * Retrieves the size desired for the avatar. Avatars are assumed to be square, so the size is used for both the
     * height and the width.
     *
     * @return the size
     */
    public int getSize() {
        return size;
    }

    /**
     * Retrieves a flag indicating whether avatar URLs should use HTTPS. If the flag is {@code true}, the returned URLs
     * will use HTTPS; otherwise, they will use HTTP.
     *
     * @return the scheme
     */
    public boolean isSecure() {
        return secure;
    }

    /**
     * Retrieve the URL using the configured base URL. This flag is <em>only</em> applicable to avatar sources
     * which serve from the current application. Avatars hosted remotely ignore this flag
     * @return {@code true} if the URL should use the configured base URL. {@code false} otherwise
     */
    public boolean isUseConfigured() {
        return useConfigured;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }

        if (o == null || getClass() != o.getClass()) {
            return false;
        }

        AvatarRequest that = (AvatarRequest) o;

        return size == that.size && secure == that.secure && useConfigured == that.useConfigured;
    }

    @Override
    public int hashCode() {
        return Objects.hash(size, secure, useConfigured);
    }
}
