package com.atlassian.bitbucket.auth;

import javax.annotation.Nonnull;

/**
 * Controls whether remember-me authentication is disabled, always performed or only performed when a checkbox is
 * checked on the login form.
 */
public enum RememberMeMode {

    /**
     * Remember-me authentication is enabled and always performed on form-based login. The 'Remember my login' checkbox
     * is not displayed on the login form.
     */
    ALWAYS("always"),
    /**
     * Remember-me authentication is enabled but only performed when a user checks the 'Remember my login' checkbox on
     * the login form.
     */
    OPTIONAL("optional"),
    /**
     * Remember-me authentication is disabled. No remember-me cookies are returned. Any provided remember-me cookies
     * are not accepted for authentication. The 'Remember my login' checkbox is not displayed on the login form.
     */
    NEVER("never");

    private final String id;

    private RememberMeMode(String id) {
        this.id = id;
    }

    @Nonnull
    public String getId() {
        return id;
    }

    @Nonnull
    public static RememberMeMode fromId(@Nonnull String id) {
        for (RememberMeMode value : values()) {
            if (value.getId().equalsIgnoreCase(id)) {
                return value;
            }
        }

        throw new IllegalArgumentException("No RememberMeMode is associated with ID [" + id + "]");
    }
}
