package com.atlassian.bitbucket.attribute;

import com.atlassian.bitbucket.property.PropertyMap;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.*;

import static java.util.Objects.requireNonNull;

/**
 * Mutable collection of attributes mapped by string keys. Allows for storing any object as a property value.
 *
 * @see PropertyMap
 * @since 5.13
 */
public class AttributeMap implements Map<String, Object> {

    private final Map<String, Object> delegate;

    AttributeMap(Map<String, Object> delegate) {
        this.delegate = delegate;
    }

    public AttributeMap() {
        this(new HashMap<>());
    }

    @Nullable
    public <T> T getAs(@Nonnull String key, @Nonnull Class<T> expectedType) {
        requireNonNull(key, "key");
        requireNonNull(expectedType, "expectedType");

        return expectedType.cast(internalGet(key));
    }

    /**
     * Convenience method to return the property value as an instance of {@code Iterable} with element type {@code E}.
     *
     * @param key property key
     * @param iterableType class representing the expected type of the iterable
     * @param elementType class representing the expected type of the elements
     * @return property value, as an instance of {@code I}, or {@code null}, if this property map does not contain a
     * property with key {@code key}
     * @throws ClassCastException if the property exists, but is not of expected type {@code iterableType}
     */
    @Nullable
    public <E, I extends Iterable<E>> I getAs(@Nonnull String key, @Nonnull Class<I> iterableType,
                                              @Nonnull Class<E> elementType) {
        requireNonNull(key, "key");
        requireNonNull(iterableType, "iterableType");
        requireNonNull(elementType, "elementType");

        return iterableType.cast(get(key));
    }

    //region Map API
    @Override
    public void clear() {
        delegate.clear();
    }

    @Override
    public boolean containsKey(Object key) {
        return delegate.containsKey(key);
    }

    @Override
    public boolean containsValue(Object value) {
        return delegate.containsValue(value);
    }

    @Nonnull
    @Override
    public Set<Entry<String, Object>> entrySet() {
        return delegate.entrySet();
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        AttributeMap that = (AttributeMap) o;
        return Objects.equals(delegate, that.delegate);
    }

    @Override
    public Object get(Object key) {
        return internalGet(key);
    }

    @Override
    public int hashCode() {
        return delegate.hashCode();
    }

    @Override
    public boolean isEmpty() {
        return delegate.isEmpty();
    }

    @Nonnull
    @Override
    public Set<String> keySet() {
        return delegate.keySet();
    }

    @Override
    public Object put(String key, Object value) {
        return delegate.put(key, value);
    }

    @Override
    public void putAll(Map<? extends String, ?> m) {
        delegate.putAll(m);
    }

    @Override
    public Object remove(Object key) {
        return delegate.remove(key);
    }

    @Override
    public int size() {
        return delegate.size();
    }

    @Nonnull
    @Override
    public Collection<Object> values() {
        return delegate.values();
    }
    //endregion

    private Object internalGet(@Nonnull Object key) {
        return delegate.get(key);
    }
}
