package com.atlassian.application.host.plugin;

import com.atlassian.application.api.ApplicationKey;
import com.atlassian.fugue.Option;
import com.atlassian.plugin.Plugin;
import com.atlassian.plugin.PluginParseException;
import com.atlassian.plugin.descriptors.AbstractModuleDescriptor;
import com.atlassian.plugin.module.ModuleFactory;
import com.google.common.base.Function;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.StringUtils;
import org.dom4j.Element;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.Collections;
import java.util.List;
import java.util.Set;
import javax.annotation.Nullable;

/**
 * A plugin point to define {@link com.atlassian.application.host.plugin.PluginApplicationMetaData}. For example:
 *
 * <pre>
 * {@code
 *  <application key="completeModule" name="ModuleDescriptorName">
 *      <applicationKey>com.atlassian.jira.platform</applicationKey>
 *      <applicationName>Test Product</applicationName>
 *      <applicationDescriptionKey>some.key</applicationDescriptionKey>
 *      <configURI>/configureMe</configURI>
 *      <postInstallURI>/postInstall</postInstallURI>
 *      <postUpdateURI>/postUpdate/actions.do</postUpdateURI>
 *      <productHelpServerSpaceURI>help-space-010</productHelpServerSpaceURI>
 *      <productHelpCloudSpaceURI>HELPCLOUDSPACE</productHelpCloudSpaceURI>
 *      <userCountKey>other.key</userCountKey>
 *      <applicationPlugins>
 *          <plugin>one</plugin>
 *          <plugin>two</plugin>
 *          <plugin>three</plugin>
 *      </applicationPlugins>
 *      <utilityPlugins>
 *          <plugin>two</plugin>
 *          <plugin>three</plugin>
 *          <plugin>four</plugin>
 *      </utilityPlugins>
 *      <defaultGroup>jira-testers</defaultGroup>
 *  </application>
 * }
 * </pre>
 *
 * The {@code applicationKey}, {@code applicationName} and {@code applicationDescriptionKey} attributes are required.
 * All other attributes are optional.
 * 
 * @since v1.0
 */
public final class DefaultApplicationMetaDataModuleDescriptor
        extends AbstractModuleDescriptor<PluginApplicationMetaData> implements ApplicationMetaDataModuleDescriptor
{
    private volatile PluginApplicationMetaData metaData;

    /**
     * Create a new module descriptor.
     *
     * @param moduleFactory the factory used to create modules. Not used in this class.
     */
    public DefaultApplicationMetaDataModuleDescriptor(final ModuleFactory moduleFactory)
    {
        super(moduleFactory);
    }

    @Override
    public void init(final Plugin plugin, final Element element) throws PluginParseException
    {
        super.init(plugin, element);

        //You can't access the OSGi Headers from the plugin as this stage. This means that we have
        //to lazy load the build date.

        final PluginApplicationMetaDataBuilder builder = new PluginApplicationMetaDataBuilder();
        builder.key(parseKey(element))
            .descriptionKey(getRequiredElement(element, Elements.DESCRIPTION))
            .userCountKey(getRequiredElement(element, Elements.USER_COUNT))
            .name(getRequiredElement(element, Elements.NAME))
            .configURI(getOptionalURI(element, Elements.CONFIG_URI))
            .postInstallURI(getOptionalURI(element, Elements.POST_INSTALL_URI))
            .postUpdateURI(getOptionalURI(element, Elements.POST_UPDATE_URI))
            .productHelpServerSpaceURI(getOptionalURI(element, Elements.PRODUCT_HELP_SERVER_SPACE_URI))
            .productHelpCloudSpaceURI(getOptionalURI(element, Elements.PRODUCT_HELP_CLOUD_SPACE_URI))
            .definitionModuleKey(getCompleteKey())
            .primaryPlugin(plugin)
            .applicationPlugins(parsePlugins(element, Elements.APPLICATION_PLUGINS))
            .defaultGroup(getRequiredElement(element, Elements.DEFAULT_GROUP))
            .utilityPlugins(parsePlugins(element, Elements.UTILITY_PLUGINS));

        metaData = builder.build();
    }

    @Override
    public PluginApplicationMetaData getModule()
    {
        return metaData;
    }

    private static ApplicationKey parseKey(final Element element)
    {
        final String key = getRequiredElement(element, Elements.KEY);
        try
        {
            return ApplicationKey.valueOf(key);
        }
        catch (final IllegalArgumentException e)
        {
            throw new PluginParseException(String.format("productKey '%s' is not valid.", key), e);
        }
    }

    private static Option<URI> getOptionalURI(final Element root, final String name)
    {
        return getOptionalElement(root, name).map(new Function<String, URI>()
        {
            @Override
            public URI apply(@Nullable final String input)
            {
                try
                {
                    return new URI(input);
                }
                catch (final URISyntaxException e)
                {
                    throw new PluginParseException(String.format("The value '%s' in element '%s' is not a valid URI.",
                            input, name), e);
                }
            }
        });
    }

    private static String getRequiredElement(final Element root, final String name)
    {
        final Element productKey = root.element(name);
        if (productKey == null)
        {
            throw new PluginParseException(String.format("Element '%s' is required.", name));
        }
        final String value = StringUtils.stripToNull(productKey.getText());
        if (value == null)
        {
            throw new PluginParseException(String.format("Element '%s' must have a value.", name));
        }
        return value;
    }

    private static Option<String> getOptionalElement(final Element root, final String name)
    {
        final Element element = root.element(name);
        if (element == null)
        {
            return Option.none();
        }
        final String value = StringUtils.stripToNull(element.getText());
        if (value == null)
        {
            return Option.none();
        }
        return Option.some(value);
    }

    private static Iterable<String> parsePlugins(final Element root, final String name)
    {
        final Element plugins = root.element(name);
        if (plugins == null)
        {
            return Collections.emptyList();
        }
        else
        {
            @SuppressWarnings ("unchecked")
            final List<Element> pluginElements = (List<Element>) plugins.elements(Elements.PLUGIN);
            final Set<String> keys = Sets.newLinkedHashSet();

            for (Element pluginElement : pluginElements)
            {
                final String pluginKey = StringUtils.stripToNull(pluginElement.getText());
                if (pluginKey == null)
                {
                    throw new PluginParseException(String.format("Plugin '%s' element must have a value.", name));
                }
                keys.add(pluginKey);
            }
            return keys;
        }
    }

    @Override
    public ApplicationKey getApplicationKey()
    {
        return getModule().getKey();
    }

    /**
     * Constants for the configuration XML elements.
     *
     * @since 1.0
     */
    private static final class Elements
    {
        private static final String KEY = "applicationKey";
        private static final String NAME = "applicationName";
        private static final String CONFIG_URI = "configURI";
        private static final String POST_INSTALL_URI = "postInstallURI";
        private static final String POST_UPDATE_URI = "postUpdateURI";
        private static final String PRODUCT_HELP_SERVER_SPACE_URI = "productHelpServerSpaceURI";
        private static final String PRODUCT_HELP_CLOUD_SPACE_URI = "productHelpCloudSpaceURI";
        private static final String PLUGIN = "plugin";
        private static final String APPLICATION_PLUGINS = "applicationPlugins";
        private static final String UTILITY_PLUGINS = "utilityPlugins";
        private static final String DESCRIPTION = "applicationDescriptionKey";
        private static final String USER_COUNT = "userCountKey";
        private static final String DEFAULT_GROUP = "defaultGroup";

        private Elements()
        {
        }
    }
}
