package com.atlassian.application.api;

import com.atlassian.sal.api.license.SingleProductLicenseDetailsView;
import io.atlassian.fugue.Option;
import org.joda.time.DateTime;

import java.net.URI;

/**
 * Represents an Application (e.g. Service Desk) installed on top of the platform (e.g. JIRA).
 *
 * <p>An application may be implemented as a group of plugins {@link com.atlassian.application.api.PluginApplication}
 * or a fundamental part of the platform {@link com.atlassian.application.api.PlatformApplication} or by some other
 * yet to be determined means.
 *
 * <p>None of the parameters or return values can be {@code null} (unless noted otherwise).
 *
 * @since v1.0
 */
public interface Application
{
    /**
     * Return the {@link com.atlassian.application.api.ApplicationKey} associated with the application.
     *
     * @return the {@link com.atlassian.application.api.ApplicationKey} associated with the application.
     */
    ApplicationKey getKey();

    /**
     * Get the name of the application.
     *
     * @return the name of the application. Must not be internationalised.
     */
    String getName();

    /**
     * Get the description of the application.
     *
     * <p>The description should be internationalised for the calling user.
     *
     * @return the internationalised description for the calling user.
     */
    String getDescription();

    /**
     * Get the version of the application.
     *
     * <p>In the case of plugin applications, this will be the version of the primary plugin.
     *
     * @return the version of the application.
     */
    String getVersion();

    /**
     * Returns the internationalised description for the passed number of users. For example,
     * JIRA Service Desk might return "agent" and "agents" depending on the {@code count}.
     *
     * <p> The returned value is expected to be in lowercase and internationalised for the calling user.
     *
     * @param count the number of users to generate the description for. {@link Option#none} can be passed to get the
     * description of an unlimited number of users.
     * @return the internationalised name for the passed number of users.
     */
    String getUserCountDescription(Option<Integer> count);

    /**
     * Get the relative URI for the configuration of the application. The context path must not be included.
     *
     * <p>{@link io.atlassian.fugue.Option#none()} can be returned to indicate that there is no configuration path.
     *
     * @return the relative URI to the configuration page minus the context path or
     * {@link io.atlassian.fugue.Option#none()} if no such path exists.
     */
    Option<URI> getConfigurationURI();

    /**
     * Get the relative URI to show after the application is installed. The context path must not be included.
     *
     * <p>{@link io.atlassian.fugue.Option#none()} can be returned to indicate that there is no path.
     *
     * @return the relative URI to the post install page minus the context path or
     * {@link io.atlassian.fugue.Option#none()} if no such path exists.
     */
    Option<URI> getPostInstallURI();

    /**
     * Get the relative URI to show after the application is updated. The context path must not be included.
     *
     * <p>{@link io.atlassian.fugue.Option#none()} can be returned to indicate that there is no path.
     *
     * @return the relative URI to the post update page minus the context path or
     * {@link io.atlassian.fugue.Option#none()} if no such path exists.
     */
    Option<URI> getPostUpdateURI();

    /**
     * Returns relative URI to use as target space for product user helplinks.
     *
     * <p>{@link io.atlassian.fugue.Option#none()} can be returned to indicate that there is no server help space.
     *
     * @return the server help space to point product user helplinks to
     * {@link io.atlassian.fugue.Option#none()} if no such space exists.
     */
    Option<URI> getProductHelpServerSpaceURI();

    /**
     * Returns relative URI to use as target space for product user helplinks in cloud.
     *
     * <p>{@link io.atlassian.fugue.Option#none()} can be returned to indicate that there is no cloud help space.
     *
     * @return the cloud help space to point product user helplinks to
     * {@link io.atlassian.fugue.Option#none()} if no such space exists.
     */
    Option<URI> getProductHelpCloudSpaceURI();

    /**
     * Return the build date of the application.
     *
     * @return the build date of the application.
     */
    DateTime buildDate();

    /**
     * Return the license associated with the application. {@link io.atlassian.fugue.Option#none()} will be returned
     * if no such license exists (aka the application is unlicensed).
     *
     * @return the license associated with application or {@link io.atlassian.fugue.Option#none()} if no such
     * license exists.
     */
    Option<SingleProductLicenseDetailsView> getLicense();

    /**
     * Return the {@link com.atlassian.application.api.ApplicationAccess} associated with the application.
     *
     * @return the {@code ApplicationAccess} associated with the Application.
     */
    ApplicationAccess getAccess();

    /**
     * Get the default user group defined for the application.
     *
     * @return the default user group of the application.
     */
    String getDefaultGroup();

    /**
     * Clear (remove) application's configuration.
     *
     * This is typically implemented by the host system and should be a no-op when not implemented.
     */
    void clearConfiguration();
}
