package com.atlassian.plugins.codegen;

import io.atlassian.fugue.Option;

import static io.atlassian.fugue.Option.none;
import static io.atlassian.fugue.Option.some;
import static java.util.Objects.requireNonNull;

/**
 * Simple wrapper for a Maven artifact ID with optional group ID.
 */
public final class ArtifactId {
    /**
     * Factory method for no {@code groupId}.
     *
     * @param artifactId the Maven artifact id
     * @return a new instance
     */
    public static ArtifactId artifactId(final String artifactId) {
        return new ArtifactId(none(String.class), artifactId);
    }

    /**
     * Factory method for both a {@code groupId} and an {@code artifactId}.
     *
     * @param groupId    the Maven group id
     * @param artifactId the Maven artifact id
     * @return a new instance
     */
    public static ArtifactId artifactId(String groupId, String artifactId) {
        return new ArtifactId(some(groupId), artifactId);
    }

    /**
     * Factory method for an optional {@code groupId}.
     *
     * @param groupId    the Maven group id, if any
     * @param artifactId the Maven artifact id
     * @return a new instance
     */
    public static ArtifactId artifactId(Option<String> groupId, String artifactId) {
        return new ArtifactId(groupId, artifactId);
    }

    private final Option<String> groupId;
    private final String artifactIdStr;

    private ArtifactId(final Option<String> groupId, final String artifactId) {
        this.groupId = requireNonNull(groupId, "groupId");
        this.artifactIdStr = requireNonNull(artifactId, "artifactId");
    }

    public Option<String> getGroupId() {
        return groupId;
    }

    public String getArtifactId() {
        return artifactIdStr;
    }

    public String getCombinedId() {
        final String prefix = groupId.map(g -> g + ":").getOrElse("");
        return prefix + artifactIdStr;
    }

    @Override
    public String toString() {
        return getCombinedId();
    }

    @Override
    public boolean equals(Object other) {
        if (other instanceof ArtifactId) {
            ArtifactId c = (ArtifactId) other;
            return groupId.equals(c.groupId) && artifactIdStr.equals(c.artifactIdStr);
        }
        return false;
    }

    @Override
    public int hashCode() {
        return getCombinedId().hashCode();
    }
}
