package com.atlassian.adf.util;

import com.atlassian.annotations.Internal;
import com.atlassian.annotations.nullability.ReturnValuesAreNonnullByDefault;

import java.util.Arrays;
import java.util.Iterator;
import java.util.List;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import java.util.stream.StreamSupport;

@Internal
@ReturnValuesAreNonnullByDefault
public abstract class Functions {
    private Functions() {
        // static-only class
    }

    /**
     * Convenience wrapper to turn a {@code Consumer<T>} into a {@code Function<T, Void>}.
     * It is common to need to do this when working with lambdas, and unfortunately Java doesn't know
     * how to work this out for itself.
     *
     * @param <T>    the inferred input parameter type for the original {@code Consumer}
     * @param effect the consumer to be reimagined as a {@code Void}-returning function
     * @return the resulting function that returns {@code Void}
     */
    public static <T> Function<T, Void> voidFn(Consumer<T> effect) {
        return t -> {
            effect.accept(t);
            return null;
        };
    }

    /**
     * Given an array, map each element from the array to a new value and return the resulting values as a list.
     *
     * @param <T>    the inferred input type
     * @param <U>    the inferred output type
     * @param source the array of input values
     * @param mapper the mapping function to apply to each input value; must not return {@code null}
     * @return an (unmodifiable) list of the resulting values
     */
    public static <T, U> List<U> mapToList(T[] source, Function<? super T, ? extends U> mapper) {
        return mapToList(Arrays.stream(source), mapper);
    }

    /**
     * Given an iterable collection, map each element from the collection to a new value and return the
     * resulting values as a list.
     *
     * @param <T>    the inferred input type
     * @param <U>    the inferred output type
     * @param source the collection of input values
     * @param mapper the mapping function to apply to each input value; must not return {@code null}
     * @return an (unmodifiable) list of the resulting values
     */
    public static <T, U> List<U> mapToList(Iterable<T> source, Function<? super T, ? extends U> mapper) {
        return mapToList(StreamSupport.stream(source.spliterator(), false), mapper);
    }

    /**
     * Given a stream, map each element from it to a new value and return the resulting values as a list.
     *
     * @param <T>    the inferred input type
     * @param <U>    the inferred output type
     * @param source the stream of input values
     * @param mapper the mapping function to apply to each input value; must not return {@code null}
     * @return an (unmodifiable) list of the resulting values
     */
    public static <T, U> List<U> mapToList(Stream<T> source, Function<? super T, ? extends U> mapper) {
        return source.map(mapper)
                .collect(Collectors.toUnmodifiableList());
    }

    /**
     * Iterates the {@code source} collection, calling {@code effect} for each element with calls to the
     * {@code joiner} between them. If the {@code source} collection is empty, there is no effect.
     * <p>
     * For example, if {@code source} is {@code List.of("foo", "bar", "baz")}, then the sequence of calls
     * will be:
     * <pre><code>
     *     effect.accept("foo");
     *     joiner.run();
     *     effect.accept("bar");
     *     joiner.run();
     *     effect.accept("baz");
     * </code></pre>
     *
     * @param <T>    the inferred element type supplied by {@code source}
     * @param source supplies the elements to iterate over
     * @param effect what to do with each element from the collection
     * @param joiner what to do between any two elements from the collection
     * @return {@code true} if the {@code source} collection supplied at least one element; {@code false} if
     * it was empty
     */
    public static <T> boolean iterateJoined(Iterable<T> source, Consumer<? super T> effect, Runnable joiner) {
        Iterator<T> iter = source.iterator();
        if (!iter.hasNext()) return false;

        effect.accept(iter.next());
        while (iter.hasNext()) {
            joiner.run();
            effect.accept(iter.next());
        }
        return true;
    }

}
