package com.atlassian.adf.util;

import com.atlassian.adf.model.ex.AdfException;
import com.atlassian.annotations.Internal;
import com.atlassian.annotations.nullability.ReturnValuesAreNonnullByDefault;

import javax.annotation.Nullable;
import java.util.Arrays;
import java.util.Collections;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;

import static java.util.Objects.requireNonNull;

/**
 * Extracts a lookup map from an enumerated type.
 *
 * @param <E> the enumerated type to be parsed
 */
@Internal
@ReturnValuesAreNonnullByDefault
public class EnumParser<E extends Enum<E>> {
    private final String enumName;
    private final Map<String, E> lookup;

    /**
     * Create a parser for an enumerated type.
     *
     * @param enumClass    the enumerated type
     * @param keyExtractor a function to extract the lookup key from an instance of {@code enumClass}. The
     *                     {@code keyExtractor} must return a distinct string for each of the enumerated values.
     */
    public EnumParser(Class<E> enumClass, Function<E, String> keyExtractor) {
        this.enumName = nameOf(enumClass);
        this.lookup = Collections.unmodifiableMap(
                Arrays.stream(enumClass.getEnumConstants())
                        .collect(Collectors.toMap(
                                keyExtractor,
                                Function.identity()
                        ))
        );
    }

    /**
     * Find the enumerated value for which the {@code keyExtractor} return {@code key} as the value.
     *
     * @param key the key value to resolve
     * @return the matching enumerated value
     * @throws AdfException.UnsupportedEnumValue if {@code key} is {@code null} or does not match any of the
     *                                           extracted key values
     */
    public E parse(@Nullable String key) {
        if (key != null) {
            E found = lookup.get(key);
            if (found != null) {
                return found;
            }
        }
        throw new AdfException.UnsupportedEnumValue(enumName, key);
    }

    /**
     * Find the enumerated value for which the {@code keyExtractor} return {@code key} as the value,
     * with {@code null} values tolerated.
     *
     * @param key the key value to resolve, or {@code null}
     * @return the matching enumerated value, or {@code null} if {@code key} was {@code null}
     * @throws IllegalArgumentException if {@code key} does not match any of the extracted key values
     */
    @Nullable
    public E parseAllowNull(@Nullable String key) {
        return (key != null) ? parse(key) : null;
    }

    private static String nameOf(Class<?> enumClass) {
        requireNonNull(enumClass, "enumClass");
        String enumName = requireNonNull(enumClass.getSimpleName(), "enumName");
        if (enumName.isEmpty()) {
            throw new IllegalStateException("How did we get a blank enumName?!");
        }
        return Character.toLowerCase(enumName.charAt(0)) + enumName.substring(1);
    }
}
