package com.atlassian.adf.util;

import com.atlassian.annotations.Internal;
import com.atlassian.annotations.nullability.ReturnValuesAreNonnullByDefault;

import java.time.Instant;
import java.time.LocalDate;
import java.time.ZonedDateTime;
import java.time.format.DateTimeParseException;

import static java.time.ZoneOffset.UTC;

/**
 * From {@code packages/editor/editor-common/src/utils/date.ts} in {@code atlassian-frontend}.
 */
// The TypeScript code relies on `new Date(...)` to do the heavy lifting.
// In Java land, we have to work a bit harder to get the value in a standard form.
@Internal
@ReturnValuesAreNonnullByDefault
public class DateUtil {
    // example: "2018-01-23"
    public static String timestampToIsoFormat(String timestamp) {
        LocalDate localDate = parseDate(timestamp);
        StringBuilder sb = new StringBuilder()
                .append(localDate.getYear())
                .append('-');

        int month = localDate.getMonthValue();
        if (month < 10) sb.append('0');
        sb.append(month).append('-');

        int day = localDate.getDayOfMonth();
        if (day < 10) sb.append('0');
        return sb.append(day).toString();
    }

    private static LocalDate parseDate(String timestamp) {
        if (isPureNumeric(timestamp)) {
            long millis = Long.parseLong(timestamp);
            Instant instant = Instant.ofEpochMilli(millis);
            return LocalDate.ofInstant(instant, UTC);
        }

        try {
            return ZonedDateTime.parse(timestamp)
                    .withZoneSameInstant(UTC)
                    .toLocalDate();
        } catch (DateTimeParseException ignore) {
            return LocalDate.parse(timestamp);
        }
    }

    private static boolean isPureNumeric(String s) {
        for (int i = 0; i < s.length(); ++i) {
            if (!Char.isDigit(s.charAt(i))) return false;
        }
        return true;
    }
}
