package com.atlassian.adf.model.node;

import com.atlassian.adf.model.node.type.DocContent;
import com.atlassian.adf.model.node.type.LayoutColumnContent;
import com.atlassian.adf.model.node.type.NonNestableBlockContent;
import com.atlassian.adf.model.node.type.PanelContent;
import com.atlassian.adf.model.node.type.TableCellContent;
import com.atlassian.adf.util.EnumParser;
import com.atlassian.adf.util.Factory;

import java.util.Map;
import java.util.Objects;
import java.util.stream.Stream;

import static com.atlassian.adf.model.node.Paragraph.p;
import static com.atlassian.adf.util.FieldMap.map;
import static com.atlassian.adf.util.ParserSupport.checkType;
import static com.atlassian.adf.util.ParserSupport.getAttr;
import static com.atlassian.adf.util.ParserSupport.getAttrOrThrow;
import static java.util.Objects.requireNonNull;

/**
 * A container that highlights content as additional information, a warning, or the result of an action.
 * <h2>Example</h2>
 * <h3>Java</h3>
 * <pre>
 * Panel.{@link #info(PanelContent) info}(
 *         {@link Paragraph#p(String) p}("Hello world")
 * )
 * </pre>
 * <h3>ADF</h3>
 * <pre>{@code
 *   {
 *     "type": "panel",
 *     "attrs": {
 *       "panelType": "info"
 *     },
 *     "content": [
 *       {
 *         "type": "paragraph",
 *         "content": [
 *           {
 *             "type": "text",
 *             "text": "Hello world"
 *           }
 *         ]
 *       }
 *     ]
 *   }
 * }</pre>
 * <h3>Result</h3>
 * <div style="color: rgb(23, 43, 77); background-color: #ffffff;">
 * <div style="background-color: rgb(222, 235, 255); border-radius: 3px; padding: 16px; width: 75%;">&#x2139;&#xFE0F;
 * Hello world
 * </div>
 * </div>
 * <p>
 * Note: This example output is not using AtlasKit to render the panel, so while it gives a vague impression
 * of what a "panel" is, it does not faithfully reproduce the actual presentation in Atlassian products.
 *
 * @see <a href="https://developer.atlassian.com/cloud/jira/platform/apis/document/nodes/panel/">Node - panel</a>
 */
@SuppressWarnings("UnnecessaryUnicodeEscape")
public class Panel
        extends AbstractContentNode<Panel, PanelContent>
        implements DocContent, LayoutColumnContent, NonNestableBlockContent, TableCellContent {

    static Factory<Panel> FACTORY = new Factory<>(Type.PANEL, Panel.class, Panel::parse);

    private PanelType panelType;
    private String panelIconId;
    private String panelIcon;
    private String panelColor;
    private String panelIconText;

    private Panel(PanelType panelType) {
        this.panelColor = "";
        this.panelIcon = "";
        this.panelIconId = "";
        this.panelIconText = "";
        this.panelType = panelType;
    }

    @Override
    protected void validateContentNodeForAppend(PanelContent node) {
        super.validateContentNodeForAppend(node);
        if (node instanceof Paragraph) {
            ((Paragraph) node).disableMarks(this);
        } else if (node instanceof Heading) {
            ((Heading) node).disableMarks(this);
        }
    }

    public static Partial.NeedsPanelType panel() {
        return new Partial.NeedsPanelType();
    }

    public static Panel info(String content) {
        return panel().info().content(p(content));
    }

    public static Panel info(String... content) {
        return panel().info().content(p(content));
    }

    public static Panel info(PanelContent content) {
        return panel().info().content(content);
    }

    public static Panel info(PanelContent... content) {
        return panel().info().content(content);
    }

    public static Panel info(Iterable<? extends PanelContent> content) {
        return panel().info().content(content);
    }

    public static Panel info(Stream<? extends PanelContent> content) {
        return panel().info().content(content);
    }

    public static Panel tip(String content) {
        return panel().tip().content(p(content));
    }

    public static Panel tip(String... content) {
        return panel().tip().content(p(content));
    }

    public static Panel tip(PanelContent content) {
        return panel().tip().content(content);
    }

    public static Panel tip(PanelContent... content) {
        return panel().tip().content(content);
    }

    public static Panel tip(Iterable<? extends PanelContent> content) {
        return panel().tip().content(content);
    }

    public static Panel tip(Stream<? extends PanelContent> content) {
        return panel().tip().content(content);
    }

    public static Panel note(String content) {
        return panel().note().content(p(content));
    }

    public static Panel note(String... content) {
        return panel().note().content(p(content));
    }

    public static Panel note(PanelContent content) {
        return panel().note().content(content);
    }

    public static Panel note(PanelContent... content) {
        return panel().note().content(content);
    }

    public static Panel note(Iterable<? extends PanelContent> content) {
        return panel().note().content(content);
    }

    public static Panel note(Stream<? extends PanelContent> content) {
        return panel().note().content(content);
    }

    public static Panel warning(String content) {
        return panel().warning().content(p(content));
    }

    public static Panel warning(String... content) {
        return panel().warning().content(p(content));
    }

    public static Panel warning(PanelContent content) {
        return panel().warning().content(content);
    }

    public static Panel warning(PanelContent... content) {
        return panel().warning().content(content);
    }

    public static Panel warning(Iterable<? extends PanelContent> content) {
        return panel().warning().content(content);
    }

    public static Panel warning(Stream<? extends PanelContent> content) {
        return panel().warning().content(content);
    }

    public static Panel success(String content) {
        return panel().success().content(p(content));
    }

    public static Panel success(String... content) {
        return panel().success().content(p(content));
    }

    public static Panel success(PanelContent content) {
        return panel().success().content(content);
    }

    public static Panel success(PanelContent... content) {
        return panel().success().content(content);
    }

    public static Panel success(Iterable<? extends PanelContent> content) {
        return panel().success().content(content);
    }

    public static Panel success(Stream<? extends PanelContent> content) {
        return panel().success().content(content);
    }

    public static Panel error(String content) {
        return panel().error().content(p(content));
    }

    public static Panel error(String... content) {
        return panel().error().content(p(content));
    }

    public static Panel error(PanelContent content) {
        return panel().error().content(content);
    }

    public static Panel error(PanelContent... content) {
        return panel().error().content(content);
    }

    public static Panel error(Iterable<? extends PanelContent> content) {
        return panel().error().content(content);
    }

    public static Panel error(Stream<? extends PanelContent> content) {
        return panel().error().content(content);
    }

    public static Panel custom(String content) {
        return panel().custom().content(p(content));
    }

    public static Panel custom(String... content) {
        return panel().custom().content(p(content));
    }

    public static Panel custom(PanelContent content) {
        return panel().custom().content(content);
    }

    public static Panel custom(PanelContent... content) {
        return panel().custom().content(content);
    }

    public static Panel custom(Iterable<? extends PanelContent> content) {
        return panel().custom().content(content);
    }

    public static Panel custom(Stream<? extends PanelContent> content) {
        return panel().custom().content(content);
    }

    /**
     * @return a new, empty panel of the given type. At least one content item must be added to make it valid.
     */
    public static Panel panel(String panelType) {
        requireNonNull(panelType, "panelType");
        return panel(PanelType.PARSER.parse(panelType));
    }

    /**
     * @return a new, empty panel of the given type. At least one content item must be added to make it valid.
     */
    public static Panel panel(PanelType panelType) {
        requireNonNull(panelType, "panelType");

        // If the panel type is custom, we need to add the default attributes
        if (panelType == PanelType.CUSTOM) {
            return panel().custom();
        }

        return new Panel(panelType);
    }

    /**
     * @return a new panel with the given type and content
     */
    public static Panel panel(String panelType, PanelContent content) {
        return panel(panelType).content(content);
    }

    /**
     * @return a new panel with the given type and content
     */
    public static Panel panel(String panelType, PanelContent... content) {
        return panel(panelType).content(content);
    }

    /**
     * @return a new panel with the given type and content
     */
    public static Panel panel(String panelType, Iterable<? extends PanelContent> content) {
        return panel(panelType).content(content);
    }

    /**
     * @return a new panel with the given type and content
     */
    public static Panel panel(String panelType, Stream<? extends PanelContent> content) {
        return panel(panelType).content(content);
    }

    /**
     * @return a new panel with the given type and content
     */
    public static Panel panel(PanelType panelType, PanelContent content) {
        return panel(panelType).content(content);
    }

    /**
     * @return a new panel with the given type and content
     */
    public static Panel panel(PanelType panelType, PanelContent... content) {
        return panel(panelType).content(content);
    }

    /**
     * @return a new panel with the given type and content
     */
    public static Panel panel(PanelType panelType, Iterable<? extends PanelContent> content) {
        return panel(panelType).content(content);
    }

    /**
     * @return a new panel with the given type and content
     */
    public static Panel panel(PanelType panelType, Stream<? extends PanelContent> content) {
        return panel(panelType).content(content);
    }

    public Panel content(String content) {
        return content(p(content));
    }

    public Panel content(String... content) {
        return content(p(content));
    }

    /**
     * Return the panel's type.
     *
     * @return the panel's type.
     */
    public PanelType panelType() {
        return panelType;
    }

    public String panelIconId() {
        return panelIconId;
    }

    public String panelIcon() {
        return panelIcon;
    }

    public String panelColor() {
        return panelColor;
    }

    public String panelIconText() {
        return panelIconText;
    }

    public Panel panelType(String panelType) {
        return panelType(PanelType.PARSER.parse(panelType));
    }

    public Panel panelType(PanelType panelType) {
        this.panelType = requireNonNull(panelType, "panelType");
        return this;
    }

    public Panel panelIconId(String panelIconId) {
        this.panelIconId = requireNonNull(panelIconId, "panelIconId");
        return this;
    }

    public Panel panelIcon(String panelIcon) {
        this.panelIcon = requireNonNull(panelIcon, "panelIcon");
        return this;
    }

    public Panel panelColor(String panelColor) {
        this.panelColor = requireNonNull(panelColor, "panelColor");
        return this;
    }

    public Panel panelIconText(String panelIconText) {
        this.panelIconText = requireNonNull(panelIconText, "panelIconText");
        return this;
    }

    public Panel info() {
        return panelType(PanelType.INFO);
    }

    public Panel note() {
        return panelType(PanelType.NOTE);
    }

    public Panel tip() {
        return panelType(PanelType.TIP);
    }

    public Panel warning() {
        return panelType(PanelType.WARNING);
    }

    public Panel error() {
        return panelType(PanelType.ERROR);
    }

    public Panel success() {
        return panelType(PanelType.SUCCESS);
    }

    public Panel custom() {
        return panelType(PanelType.CUSTOM);
    }

    @Override
    public Panel copy() {
        return parse(toMap());
    }

    @Override
    public String elementType() {
        return Type.PANEL;
    }

    @Override
    protected int contentNodeHashCode() {
        return Objects.hash(panelType, panelColor, panelIcon, panelIconId, panelIconText);
    }

    @Override
    protected boolean contentNodeEquals(Panel other) {
        return this.panelType.equals(other.panelType)
                && this.panelColor.equals(other.panelColor)
                && this.panelIcon.equals(other.panelIcon)
                && this.panelIconId.equals(other.panelIconId)
                && this.panelIconText.equals(other.panelIconText);
    }

    @Override
    protected void appendContentNodeFields(ToStringHelper buf) {
        buf.appendField("panelType", panelType);
        buf.appendField("panelColor", panelColor);
        buf.appendField("panelIcon", panelIcon);
        buf.appendField("panelIconId", panelIconId);
        buf.appendField("panelIconText", panelIconText);
    }

    @Override
    protected void contentNodeValidate() {
        requireNotEmpty();
        for (PanelContent node : content) {
            if (node instanceof Paragraph) {
                ((Paragraph) node).disableMarks(this);
            } else if (node instanceof Heading) {
                ((Heading) node).disableMarks(this);
            }
        }
    }

    @Override
    public Map<String, ?> toMap() {
        requireNotEmpty();
        return mapWithType()
                .let(this::addContent)
                .add(Key.ATTRS, map()
                        .add(Attr.PANEL_TYPE, panelType.panelType())
                        .addIf(!panelColor.isEmpty(), Attr.PANEL_COLOR, () -> panelColor)
                        .addIf(!panelIcon.isEmpty(), Attr.PANEL_ICON, () -> panelIcon)
                        .addIf(!panelIconId.isEmpty(), Attr.PANEL_ICON_ID, () -> panelIconId)
                        .addIf(!panelIconText.isEmpty(), Attr.PANEL_ICON_TEXT, () -> panelIconText)
                );
    }

    private static Panel parse(Map<String, ?> map) {
        checkType(map, Type.PANEL);
        Panel parsedPanel = panel(getAttrOrThrow(map, Attr.PANEL_TYPE, String.class))
                .parseRequiredContent(map, PanelContent.class);

        getAttr(map, Attr.PANEL_COLOR, String.class).ifPresent(parsedPanel::panelColor);
        getAttr(map, Attr.PANEL_ICON, String.class).ifPresent(parsedPanel::panelIcon);
        getAttr(map, Attr.PANEL_ICON_ID, String.class).ifPresent(parsedPanel::panelIconId);
        getAttr(map, Attr.PANEL_ICON_TEXT, String.class).ifPresent(parsedPanel::panelIconText);

        return parsedPanel;
    }

    @Override
    public void appendPlainText(StringBuilder sb) {
        appendPlainTextContentJoinedWith('\n', sb);
    }

    public enum PanelType {
        INFO("info"),
        NOTE("note"),
        TIP("tip"),
        WARNING("warning"),
        ERROR("error"),
        SUCCESS("success"),
        CUSTOM("custom");

        static final EnumParser<PanelType> PARSER = new EnumParser<>(PanelType.class, PanelType::panelType);

        private final String panelType;

        PanelType(String panelType) {
            this.panelType = panelType;
        }

        public String panelType() {
            return panelType;
        }
    }

    /**
     * Default attributes that represent a {@link Panel panel} type.
     */
    public interface DEFAULT_ATTR {
        String CUSTOM_COLOR = "#E6FCFF";
        String CUSTOM_ICON = ":rainbow:";
        String CUSTOM_ICON_ID = "1f308";
        String CUSTOM_ICON_TEXT = "\uD83C\uDF08";
    }


    /**
     * Types that represent a partially constructed {@link Panel panel}.
     */
    public interface Partial {
        /**
         * This partially constructed {@code panel} still needs its {@code panelType} attribute set.
         */
        class NeedsPanelType {
            public Panel panelType(String panelType) {
                return panelType(PanelType.PARSER.parse(panelType));
            }

            public Panel panelType(PanelType panelType) {
                return new Panel(panelType);
            }

            public Panel info() {
                return new Panel(PanelType.INFO);
            }

            public Panel note() {
                return new Panel(PanelType.NOTE);
            }

            public Panel tip() {
                return new Panel(PanelType.TIP);
            }

            public Panel warning() {
                return new Panel(PanelType.WARNING);
            }

            public Panel error() {
                return new Panel(PanelType.ERROR);
            }

            public Panel success() {
                return new Panel(PanelType.SUCCESS);
            }

            public Panel custom() {
                // Passing in the default attributes for custom panels
                return new Panel(PanelType.CUSTOM)
                        .panelColor(DEFAULT_ATTR.CUSTOM_COLOR)
                        .panelIcon(DEFAULT_ATTR.CUSTOM_ICON)
                        .panelIconId(DEFAULT_ATTR.CUSTOM_ICON_ID)
                        .panelIconText(DEFAULT_ATTR.CUSTOM_ICON_TEXT);
            }

        }
    }
}
