package com.atlassian.adf.model.node;

import com.atlassian.adf.model.Documentation;
import com.atlassian.adf.model.ex.node.ListException;
import com.atlassian.adf.util.Factory;

import javax.annotation.Nullable;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Stream;

import static com.atlassian.adf.util.FieldMap.map;
import static com.atlassian.adf.util.Functions.iterateJoined;
import static com.atlassian.adf.util.ParserSupport.checkType;
import static com.atlassian.adf.util.ParserSupport.getAttrInt;

/**
 * A container for {@link ListItem list items} that produces a numbered list of them.
 * <h2>Example</h2>
 * <h3>Java</h3>
 * <pre>
 * {@link #orderedList(int, ListItem[]) orderedList}(
 *         3,
 *         {@link ListItem#li(String) li}("Hello"),
 *         {@link ListItem#li(String) li}("World")
 * );
 * </pre>
 * <h3>ADF</h3>
 * <pre>{@code
 *   {
 *     "type": "orderedList",
 *     "attrs": {
 *       "order": 3
 *     },
 *     "content": [
 *       {
 *         "type": "listItem",
 *         "content": [
 *           {
 *             "type": "paragraph",
 *             "content": [
 *               {
 *                 "type": "text",
 *                 "text": "Hello"
 *               }
 *             ]
 *           }
 *         ]
 *       },
 *       {
 *         "type": "listItem",
 *         "content": [
 *           {
 *             "type": "paragraph",
 *             "content": [
 *               {
 *                 "type": "text",
 *                 "text": "World"
 *               }
 *             ]
 *           }
 *         ]
 *       }
 *     ]
 *   }
 * }</pre>
 * <h3>Result</h3>
 * <div style="color: rgb(23, 43, 77); background-color: #ffffff;">
 * <ol type="1" start="3">
 *     <li><p>Hello</p></li>
 *     <li><p>World</p></li>
 * </ol>
 * </div>
 *
 * @see <a href="https://developer.atlassian.com/cloud/jira/platform/apis/document/nodes/orderedList/">Node - orderedList</a>
 */
@Documentation(
        state = Documentation.State.WRONG,
        date = "2023-07-26",
        comment = "the 'order' may be 0, now"
)
public class OrderedList extends AbstractListNode<OrderedList> {
    static Factory<OrderedList> FACTORY = new Factory<>(Type.ORDERED_LIST, OrderedList.class, OrderedList::parse);

    @Nullable
    private Integer order;

    private OrderedList() {
    }

    /**
     * @return a new, empty ordered list. At least one list item must be added before it will be valid.
     */
    public static OrderedList ol() {
        return new OrderedList();
    }

    /**
     * @return a new ordered list with the given contents
     */
    public static OrderedList ol(ListItem... content) {
        return ol().content(content);
    }

    /**
     * @return a new ordered list with the given contents
     */
    public static OrderedList ol(Iterable<? extends ListItem> content) {
        return ol().content(content);
    }

    /**
     * @return a new ordered list with the given contents
     */
    public static OrderedList ol(Stream<? extends ListItem> content) {
        return ol().content(content);
    }

    /**
     * @return a new, empty ordered list with the specified initial order. At least one list item must be
     * added before it will be valid.
     */
    public static OrderedList ol(int order) {
        return ol().order(order);
    }

    /**
     * @return a new ordered list with the specified initial order and contents.
     */
    public static OrderedList ol(int order, ListItem... content) {
        return ol().order(order).content(content);
    }

    /**
     * @return a new ordered list with the specified initial order and contents.
     */
    public static OrderedList ol(int order, Iterable<? extends ListItem> content) {
        return ol().order(order).content(content);
    }

    /**
     * @return a new ordered list with the specified initial order and contents.
     */
    public static OrderedList ol(int order, Stream<? extends ListItem> content) {
        return ol().order(order).content(content);
    }

    /**
     * @see #ol()
     */
    public static OrderedList orderedList() {
        return new OrderedList();
    }

    /**
     * @see #ol(ListItem[])
     */
    public static OrderedList orderedList(ListItem... content) {
        return orderedList().content(content);
    }

    /**
     * @see #ol(Iterable)
     */
    public static OrderedList orderedList(Iterable<? extends ListItem> content) {
        return orderedList().content(content);
    }

    /**
     * @see #ol(Stream)
     */
    public static OrderedList orderedList(Stream<? extends ListItem> content) {
        return orderedList().content(content);
    }

    /**
     * @see #ol(int)
     */
    public static OrderedList orderedList(int order) {
        return orderedList().order(order);
    }

    /**
     * @see #ol(int, ListItem[])
     */
    public static OrderedList orderedList(int order, ListItem... content) {
        return orderedList().order(order).content(content);
    }

    /**
     * @see #ol(int, Iterable)
     */
    public static OrderedList orderedList(int order, Iterable<? extends ListItem> content) {
        return orderedList().order(order).content(content);
    }

    /**
     * @see #ol(int, Stream)
     */
    public static OrderedList orderedList(int order, Stream<? extends ListItem> content) {
        return ol().order(order).content(content);
    }

    /**
     * Returns the ordinal of the initial value in the list, if set.
     *
     * @return the ordinal of the initial value in the list, or {@code empty()} if not set.
     */
    public Optional<Integer> order() {
        return Optional.ofNullable(order);
    }

    /**
     * Changes the starting value for the list, which otherwise defaults to starting at {@code 1}.
     *
     * @param order the positive integer to use as the starting point when numbering the list items
     * @return {@code this}
     */
    public OrderedList order(@Nullable Integer order) {
        if (order != null && order < 0) {
            throw new ListException.OrderMustNotBeNegative(order);
        }
        this.order = order;
        return this;
    }

    @Override
    public OrderedList copy() {
        return parse(toMap());
    }

    @Override
    public String elementType() {
        return Type.ORDERED_LIST;
    }

    @Override
    public Map<String, ?> toMap() {
        requireNotEmpty();
        return mapWithType()
                .let(this::addContent)
                .addIf(order != null, Key.ATTRS, () -> map(Attr.ORDER, order));
    }

    private static OrderedList parse(Map<String, ?> map) {
        checkType(map, Type.ORDERED_LIST);
        OrderedList ol = ol().parseRequiredContent(map, ListItem.class);
        getAttrInt(map, Attr.ORDER)
                .filter(x -> x >= 0)  // Silently discard invalid 'order' values
                .ifPresent(ol::order);
        return ol;
    }

    @Override
    protected boolean contentNodeEquals(OrderedList other) {
        return Objects.equals(order, other.order);
    }

    @Override
    protected int contentNodeHashCode() {
        return Objects.hashCode(order);
    }

    @Override
    protected void appendContentNodeFields(ToStringHelper buf) {
        buf.appendField("order", order);
    }

    @Override
    public void appendPlainText(StringBuilder sb) {
        if (content.isEmpty()) return;

        // Have to hide it inside an array to make it mutable by the lambda. Silly Java.
        int[] index = {order().orElse(1)};
        sb.append(index[0]).append(". ");

        iterateJoined(
                content,
                child -> child.appendPlainText(sb),
                () -> sb.append('\n')
                        .append(++index[0])
                        .append(". ")
        );
    }
}
