package com.atlassian.adf.model.node;

import com.atlassian.adf.model.Documentation;
import com.atlassian.adf.model.ex.AdfException;
import com.atlassian.adf.model.ex.node.MediaException;
import com.atlassian.adf.model.mark.Border;
import com.atlassian.adf.model.mark.Link;
import com.atlassian.adf.model.mark.Mark;
import com.atlassian.adf.model.mark.type.MediaMark;
import com.atlassian.adf.model.node.type.Marked;
import com.atlassian.adf.util.EnumParser;
import com.atlassian.adf.util.Factory;
import com.atlassian.adf.util.FieldMap;
import com.atlassian.adf.util.Fold3;

import javax.annotation.CheckReturnValue;
import javax.annotation.Nullable;
import java.net.URI;
import java.net.URL;
import java.util.*;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Stream;

import static com.atlassian.adf.model.Element.nonEmpty;
import static com.atlassian.adf.model.Element.nonNull;
import static com.atlassian.adf.model.mark.MarkParserSupport.parseMarks;
import static com.atlassian.adf.util.Cast.unsafeCast;
import static com.atlassian.adf.util.FieldMap.map;
import static com.atlassian.adf.util.Functions.voidFn;
import static com.atlassian.adf.util.ParserSupport.*;
import static java.util.Objects.requireNonNull;

/**
 * Represents a single file or link stored in media services.
 * There is also an {@link #externalMedia() external} media type which takes a URL instead
 * of Media Services information.
 * <h2>Example</h2>
 * <h3>Java</h3>
 * <pre><code>
 * Media.{@link #fileMedia(String, String) fileMedia}(
 *         "6e7c7f2c-dd7a-499c-bceb-6f32bfbf30b5",
 *         "my project files"
 * ).{@link Media#size(Number, Number) size}(200, 183);
 * </code></pre>
 * Or equivalently:
 * <pre><code>
 * Media.{@link #media() media}()
 *         .{@link Media.Partial.NeedsType#file file}()
 *         .{@link Media.Partial.FileNeedsId#id(String) id}("6e7c7f2c-dd7a-499c-bceb-6f32bfbf30b5")
 *         .{@link Media.FileOrLinkMedia#collection(String) collection}("my project files");
 *         .{@link Media#size(Number, Number) size}(200, 183);
 * </code></pre>
 *
 * <h3>ADF</h3>
 * <pre>{@code
 *   {
 *     "type": "media",
 *     "attrs": {
 *       "id": "6e7c7f2c-dd7a-499c-bceb-6f32bfbf30b5",
 *       "collection": "my project files",
 *       "type": "file",
 *       "width": 200,
 *       "height": 183
 *     }
 *   }
 * }</pre>
 * <p>
 *
 * <h3>Result</h3>
 * <img width=200 height=183
 * src="data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAgAAAAIAQMAAAD+wSzIAAAABlBMVEX///+/v7+jQ3Y5AAAADklEQVQI12P4AIX8EAgALgAD/aNpbtEAAAAASUVORK5CYII"/>
 * <p>Notes:</p>
 * <ul>
 *     <li>The media {@code collection} is being phased out, but it is currently still required by the JSON
 *         schema. This node allows the caller to fail to set it or set it to {@code null}, but will substitute
 *         the empty string {@code ""} instead, for the time being.</li>
 *     <li>This example uses a data URI to embed the image instead of retrieving the image from media services,
 *         which uses a different mechanism to supply the image content.</li>
 * </ul>
 *
 * @see <a href="https://developer.atlassian.com/cloud/jira/platform/apis/document/nodes/media/">Node - media</a>
 */
@Documentation(
        state = Documentation.State.WRONG,
        date = "2023-07-26",
        comment = "'external' type, its different requirements, and the 'link' and 'border' marks are undocumented"
)
@SuppressWarnings({"unused", "UnusedReturnValue"})
public interface Media
        extends Node,
        Marked<Media, MediaMark>,
        Fold3<Media.FileMedia, Media.LinkMedia, Media.ExternalMedia> {

    Factory<Media> FACTORY = new Factory<>(Type.MEDIA, Media.class, AbstractMedia::parse);

    /**
     * Begins the fluent construction of a media node.
     *
     * @return an incomplete media node that needs the type to be specified
     */
    @CheckReturnValue
    static Partial.NeedsType media() {
        return new Partial.NeedsType();
    }

    /**
     * Begins the fluent construction of a {@code file} media node.
     * All required fields must also be specified before the media node is valid.
     *
     * @return a partial {@code file} media node
     */
    @CheckReturnValue
    static Partial.FileNeedsId fileMedia() {
        return new Partial.FileNeedsId();
    }

    @CheckReturnValue
    static FileMedia fileMedia(String id) {
        return new FileMedia(id);
    }

    /**
     * Creates a new {@code FILE} media node with the given media ID and collection ID.
     *
     * @param id         the Media Services ID used for querying the media services API to retrieve metadata,
     *                   such as the filename. Consumers of the document should always fetch fresh metadata
     *                   using the Media API rather than cache it locally.
     * @param collection the Media Services Collection name for the media
     * @return the new {@code media} node
     */
    static FileMedia fileMedia(String id, @Nullable String collection) {
        return new FileMedia(id).collection(collection);
    }

    /**
     * Begins the fluent construction of a {@code link} media node.
     * All required fields must also be specified before the media node is valid.
     *
     * @return a partial {@code link} media node
     */
    @CheckReturnValue
    static Partial.LinkNeedsId linkMedia() {
        return new Partial.LinkNeedsId();
    }

    @CheckReturnValue
    static LinkMedia linkMedia(String id) {
        return new LinkMedia(id);
    }

    /**
     * Creates a new {@code LINK} media node with the given media ID and collection ID.
     *
     * @param id         the Media Services ID used for querying the media services API to retrieve metadata,
     *                   such as the filename. Consumers of the document should always fetch fresh metadata
     *                   using the Media API rather than cache it locally.
     * @param collection the Media Services Collection name for the media
     * @return the new {@code media} node
     */
    static LinkMedia linkMedia(String id, @Nullable String collection) {
        return new LinkMedia(id).collection(collection);
    }

    @CheckReturnValue
    static Partial.ExternalNeedsUrl externalMedia() {
        return new Partial.ExternalNeedsUrl();
    }

    static ExternalMedia externalMedia(String url) {
        return externalMedia().url(url);
    }

    static ExternalMedia externalMedia(URL url) {
        return externalMedia().url(url);
    }

    static ExternalMedia externalMedia(URI url) {
        return externalMedia().url(url);
    }

    /**
     * Returns the media node's type.
     *
     * @return the media node's type.
     */
    MediaType type();

    /**
     * Returns the media node's display width, if set.
     *
     * @return the media node's display width, or {@code empty()} if not set.
     */
    Optional<Number> width();

    /**
     * Sets the width of the media.
     * Although this attribute is optional, it must be set for the {@code media} item in a
     * {@link MediaSingle mediaSingle} node, or the media is not displayed.
     *
     * @param width the display width of the media item, in pixels; must be positive
     * @return {@code this}
     */
    Media width(@Nullable Number width);

    /**
     * Returns the media node's display height, if set.
     *
     * @return the media node's display height, or {@code empty()} if not set.
     */
    Optional<Number> height();

    /**
     * Sets the height of the media.
     * Although this attribute is optional, it must be set for the {@code media} item in a
     * {@link MediaSingle mediaSingle} node, or the media is not displayed.
     *
     * @param height the display height of the media item, in pixels; must be positive
     * @return {@code this}
     */
    Media height(@Nullable Number height);

    /**
     * Sets the width and height of the media.
     * <p>
     * This convenience method is exactly equivalent to calling
     * <code>{@link #width(Number) width}(width).{@link #height(Number) height}(height)</code>.
     *
     * @param width  as for {@link #width(Number)}
     * @param height as for {@link #height(Number)}
     * @return {@code this}
     */
    Media size(@Nullable Number width, @Nullable Number height);

    @Override
    <R> R fold(
            Function<? super FileMedia, ? extends R> ifFile,
            Function<? super LinkMedia, ? extends R> ifLink,
            Function<? super ExternalMedia, ? extends R> ifExternal
    );

    Optional<String> alt();

    Media alt(@Nullable String alt);

    Media linkMark(@Nullable Link link);

    Media linkMark(@Nullable URL url);

    Media linkMark(@Nullable String href);

    Optional<Link> linkMark();

    Media border(@Nullable Border border);

    Media border(String color);

    Media border(int size, String color);

    Optional<Border> border();

    Optional<FileMedia> file();

    Optional<LinkMedia> link();

    Optional<ExternalMedia> external();

    void ifFile(Consumer<? super FileMedia> effect);

    void ifLink(Consumer<? super LinkMedia> effect);

    void ifExternal(Consumer<? super ExternalMedia> effect);


    enum MediaType {
        FILE("file", FileMedia::parseFile),
        LINK("link", LinkMedia::parseLink),
        EXTERNAL("external", ExternalMedia::parseExternal);

        static final EnumParser<MediaType> PARSER = new EnumParser<>(MediaType.class, MediaType::mediaType);

        private final String mediaType;
        private final Function<Map<String, ?>, ? extends AbstractMedia<?>> parser;

        MediaType(String mediaType, Function<Map<String, ?>, ? extends AbstractMedia<?>> parser) {
            this.mediaType = mediaType;
            this.parser = parser;
        }

        public String mediaType() {
            return mediaType;
        }

        AbstractMedia<?> parse(Map<String, ?> map) {
            return parser.apply(map);
        }
    }

    /**
     * Types that represent a partially constructed {@link Media media}.
     */
    interface Partial {
        /**
         * Partially constructed {@code media} node still needs the specific media type set.
         */
        class NeedsType {
            NeedsType() {
            }

            /**
             * Sets this to be a {@link MediaType#FILE FILE} media node.
             */
            public FileNeedsId file() {
                return new FileNeedsId();
            }

            /**
             * Sets this to be a {@link MediaType#LINK LINK} media node.
             */
            public LinkNeedsId link() {
                return new LinkNeedsId();
            }

            /**
             * Sets this to be a {@link MediaType#EXTERNAL EXTERNAL} media node.
             */
            public ExternalNeedsUrl external() {
                return new ExternalNeedsUrl();
            }
        }

        /**
         * This partially constructed {@code media} node is a {@code file} and still needs a media {@code id}.
         */
        class FileNeedsId {
            FileNeedsId() {
            }

            @CheckReturnValue
            public FileMedia id(String id) {
                return new FileMedia(id);
            }
        }

        /**
         * This partially constructed {@code media} node is a {@code link} and still needs a media {@code id}.
         */
        class LinkNeedsId {
            LinkNeedsId() {
            }

            @CheckReturnValue
            public LinkMedia id(String id) {
                return new LinkMedia(id);
            }
        }

        /**
         * This partially constructed {@code media} node is a {@link MediaType#LINK link} and still needs a
         * {@code collection} ID.
         */
        class LinkNeedsCollection {
            private final String id;

            LinkNeedsCollection(String id) {
                this.id = nonNull(id, "id");
            }

            /**
             * Sets the collection ID on this partially constructed media node, after which it is
             * fully constructed as a valid media node.
             *
             * @param collection the collection ID for this media node
             * @return the resulting fully constructed media node, to which optional settings like the
             * {@link Media#width(Number) width} may still be set
             */
            public LinkMedia collection(@Nullable String collection) {
                return new LinkMedia(id).collection(collection);
            }
        }

        /**
         * This partially constructed {@code media} node is {@link MediaType#EXTERNAL external} and still needs a
         * {@code url}.
         */
        class ExternalNeedsUrl {

            /**
             * Sets the URI on this partially constructed media node, after which it is
             * fully constructed as a valid media node.
             *
             * @param url the target URI for the external media
             * @return the resulting fully constructed media node
             * @throws AdfException.InvalidURI if {@code url} is not a valid URI
             */
            public ExternalMedia url(String url) {
                return new ExternalMedia(cleanUri(url, "url"));
            }

            /**
             * Sets the URI on this partially constructed media node, after which it is
             * fully constructed as a valid media node.
             *
             * @param url the target URI for the external media
             * @return the resulting fully constructed media node
             * @throws AdfException.InvalidURI if {@code url} is not a valid URI
             */
            public ExternalMedia url(URL url) {
                return new ExternalMedia(cleanUri(url, "url"));
            }

            /**
             * Sets the URI on this partially constructed media node, after which it is
             * fully constructed as a valid media node.
             *
             * @param url the target URI for the external media
             * @return the resulting fully constructed media node
             */
            public ExternalMedia url(URI url) {
                return new ExternalMedia(cleanUri(url, "url"));
            }
        }
    }

    // This doesn't use AbstractMarkedNode because there's no easy way to reconcile that with `Media`
    // declaring the `Marked` interface unless we also make `Media` a parameterized type. That would
    // be a huge mess, so we'll just have to do all the mark management locally. :(
    abstract class AbstractMedia<N extends AbstractMedia<N>>
            extends AbstractNode<N>
            implements Media {

        @Nullable
        protected Number width;
        @Nullable
        protected Number height;
        @Nullable
        protected String alt;

        protected final MarkHolder<MediaMark> marks = MarkHolder.unlimited();

        @Override
        public final Class<MediaMark> markClass() {
            return MediaMark.class;
        }

        @Override
        public N mark(MediaMark mark) {
            marks.add(mark);
            return self();
        }

        public N link(Link link) {
            return mark(link);
        }

        public N link(String href) {
            return mark(Link.link(href));
        }

        @Override
        public N linkMark(@Nullable Link link) {
            marks.remove(Mark.Type.LINK);
            if (link != null) marks.add(link);
            return self();
        }

        @Override
        public N linkMark(@Nullable URL url) {
            return linkMark((url != null) ? Link.link(url) : null);
        }

        @Override
        public N linkMark(@Nullable String href) {
            return linkMark((href != null) ? Link.link(href) : null);
        }

        @Override
        public Optional<Link> linkMark() {
            return marks.get(Mark.Type.LINK)
                    .map(Link.class::cast);
        }

        @Override
        public Optional<Border> border() {
            return marks.get(Mark.Type.BORDER)
                    .map(Border.class::cast);
        }

        @Override
        public N border(@Nullable Border border) {
            marks.remove(Mark.Type.BORDER);
            if (border != null) marks.add(border);
            return self();
        }

        @Override
        public N border(@Nullable String color) {
            return border((color != null) ? Border.border().color(color) : null);
        }

        @Override
        public N border(int size, String color) {
            Border border = Border.border()
                    .size(size)
                    .color(color);
            return border(border);
        }

        @Override
        public Collection<MediaMark> marks() {
            return marks.get();
        }

        @Override
        public Set<String> markTypes() {
            return marks.getTypes();
        }

        @Override
        public <T extends MediaMark> Stream<? extends T> marks(Class<T> markClass) {
            return marks.stream(markClass);
        }

        @Override
        public Optional<MediaMark> mark(String type) {
            return marks.get(type);
        }

        @Override
        public void accept(
                Consumer<? super FileMedia> ifFile,
                Consumer<? super LinkMedia> ifLink,
                Consumer<? super ExternalMedia> ifExternal
        ) {
            fold(voidFn(ifFile), voidFn(ifLink), voidFn(ifExternal));
        }

        @Override
        public Optional<Number> width() {
            return Optional.ofNullable(width);
        }

        @Override
        public N width(@Nullable Number width) {
            if (width != null && width.doubleValue() <= 0.0) {
                throw new MediaException.WidthMustBePositive(width);
            }
            this.width = width;
            return self();
        }

        @Override
        public Optional<Number> height() {
            return Optional.ofNullable(height);
        }

        @Override
        public N height(@Nullable Number height) {
            if (height != null && height.doubleValue() <= 0.0) {
                throw new MediaException.HeightMustBePositive(height);
            }
            this.height = height;
            return self();
        }

        @Override
        public N size(@Nullable Number width, @Nullable Number height) {
            return width(width).height(height);
        }

        public Optional<String> alt() {
            return Optional.ofNullable(alt);
        }

        public N alt(@Nullable String alt) {
            this.alt = alt;
            return self();
        }

        @Override
        public void ifFile(Consumer<? super FileMedia> effect) {
            file().ifPresent(effect);
        }

        @Override
        public void ifLink(Consumer<? super LinkMedia> effect) {
            link().ifPresent(effect);
        }

        @Override
        public void ifExternal(Consumer<? super ExternalMedia> effect) {
            external().ifPresent(effect);
        }

        @Override
        public String elementType() {
            return Type.MEDIA;
        }

        @Override
        public void validate() {
        }

        @Override
        protected final boolean nodeEquals(N other) {
            return Objects.equals(alt, other.alt)
                    && numberEq(width, other.width)
                    && numberEq(height, other.height)
                    && mediaEquals(other)
                    && marks.equals(other.marks);
        }

        @Override
        protected int nodeHashCode() {
            return Objects.hash(getClass(), numberHash(width), numberHash(height), alt, mediaHashCode(), marks);
        }

        protected void appendNodeFields(ToStringHelper buf) {
            buf.appendField("mediaType", type());
            buf.appendField("width", width);
            buf.appendField("height", height);
            buf.appendField("alt", alt);
            appendMediaFields(buf);
            buf.appendMarksField(marks);
        }

        protected abstract boolean mediaEquals(N other);

        protected abstract int mediaHashCode();

        protected abstract void appendMediaFields(ToStringHelper buf);

        protected void addCommonAttrs(FieldMap attrs) {
            attrs.addIfPresent(Attr.WIDTH, width);
            attrs.addIfPresent(Attr.HEIGHT, height);
            attrs.addIfPresent(Attr.ALT, alt);
        }

        protected void parseCommonAttrs(Map<String, ?> map) {
            getAttrNumber(map, Attr.WIDTH).ifPresent(this::width);
            getAttrNumber(map, Attr.HEIGHT).ifPresent(this::height);
            getAttr(map, Attr.ALT, String.class).ifPresent(this::alt);
        }

        static Media parse(Map<String, ?> map) {
            checkType(map, Type.MEDIA);
            MediaType type = MediaType.PARSER.parse(getAttrOrThrow(map, Attr.TYPE, String.class));
            AbstractMedia<?> media = type.parse(map);
            media.parseCommonAttrs(map);
            parseMarks(map, MediaMark.class, null, media);
            return media;
        }
    }

    abstract class FileOrLinkMedia<T extends FileOrLinkMedia<T>> extends AbstractMedia<T> {
        protected String id;
        protected String collection = "";

        @Nullable
        protected String occurrenceKey;

        FileOrLinkMedia(String id) {
            this.id = validateId(id);
        }

        /**
         * Replaces the {@code id} that was set when the node was constructed.
         * <p>
         * In most cases, this library does not allow the modification of any of the values
         *
         * @param id the new {@code id} for this media node
         * @return {@code this}
         */
        public T id(String id) {
            this.id = validateId(id);
            return self();
        }

        /**
         * Replaces the {@code collection} ID that was set when the node was constructed.
         *
         * @param collection the new {@code collection} for this media node
         * @return {@code this}
         */
        public T collection(@Nullable String collection) {
            this.collection = validateCollection(collection);
            return self();
        }

        /**
         * Returns the media node's ID.
         *
         * @return the media node's ID.
         */
        public String id() {
            return id;
        }

        /**
         * Returns the media node's collection ID.
         *
         * @return the media node's collection ID.
         */
        public String collection() {
            return collection;
        }

        /**
         * Returns the media node's occurrence key, if set.
         *
         * @return the media node's occurrence key, or {@code empty()} if not set.
         */
        public Optional<String> occurrenceKey() {
            return Optional.ofNullable(occurrenceKey);
        }

        /**
         * Sets the occurrence key for this media item.
         * Although this attribute is optional, it must be set to enable deletion of files from a collection.
         *
         * @param occurrenceKey the occurrence key value
         * @return {@code this}
         */
        public T occurrenceKey(@Nullable String occurrenceKey) {
            this.occurrenceKey = validateOccurrenceKey(occurrenceKey);
            return self();
        }

        @Override
        public Optional<ExternalMedia> external() {
            return Optional.empty();
        }

        @Override
        public Map<String, ?> toMap() {
            return mapWithType()
                    .add(Key.ATTRS, map()
                            .add(Attr.TYPE, type().mediaType())
                            .add(Attr.ID, id)
                            .add(Attr.COLLECTION, collection)
                            .addIfPresent(Attr.OCCURRENCE_KEY, occurrenceKey)
                            .let(this::addCommonAttrs)
                    )
                    .let(marks::addToMap);
        }

        @Override
        protected final boolean mediaEquals(T other) {
            return id.equals(other.id)
                    && collection.equals(other.collection)
                    && Objects.equals(occurrenceKey, other.occurrenceKey);
        }

        @Override
        protected final int mediaHashCode() {
            return Objects.hash(id, collection, occurrenceKey);
        }

        @Override
        protected final void appendMediaFields(ToStringHelper buf) {
            buf.appendField("id", id);
            buf.appendField("collection", collection);
            buf.appendField("occurrenceKey", occurrenceKey);
        }

        private static String validateId(String id) {
            return nonEmpty(id, "id");
        }

        private static String validateCollection(@Nullable String collection) {
            return (collection != null) ? collection : "";
        }

        @Nullable
        private static String validateOccurrenceKey(@Nullable String occurrenceKey) {
            return (occurrenceKey != null && occurrenceKey.isEmpty()) ? null : occurrenceKey;
        }
    }

    class FileMedia extends FileOrLinkMedia<FileMedia> {
        FileMedia(String id) {
            super(id);
        }

        @Override
        public FileMedia copy() {
            return unsafeCast(parse(toMap()));
        }

        @Override
        public MediaType type() {
            return MediaType.FILE;
        }

        @Override
        public Optional<FileMedia> file() {
            return Optional.of(this);
        }

        @Override
        public Optional<LinkMedia> link() {
            return Optional.empty();
        }

        @Override
        public <R> R fold(
                Function<? super FileMedia, ? extends R> ifFile,
                Function<? super LinkMedia, ? extends R> ifLink,
                Function<? super ExternalMedia, ? extends R> ifExternal
        ) {
            return ifFile.apply(this);
        }

        static FileMedia parseFile(Map<String, ?> map) {
            String id = getAttrOrThrow(map, Attr.ID);
            String collection = getAttr(map, Attr.COLLECTION, String.class).orElse(null);
            FileMedia media = new FileMedia(id).collection(collection);
            getAttr(map, Attr.OCCURRENCE_KEY, String.class).ifPresent(media::occurrenceKey);
            return media;
        }
    }

    class LinkMedia extends FileOrLinkMedia<LinkMedia> {
        LinkMedia(String id) {
            super(id);
        }

        @Override
        public LinkMedia copy() {
            return unsafeCast(parse(toMap()));
        }

        @Override
        public MediaType type() {
            return MediaType.LINK;
        }

        @Override
        public Optional<FileMedia> file() {
            return Optional.empty();
        }

        @Override
        public Optional<LinkMedia> link() {
            return Optional.of(this);
        }

        @Override
        public <R> R fold(
                Function<? super FileMedia, ? extends R> ifFile,
                Function<? super LinkMedia, ? extends R> ifLink,
                Function<? super ExternalMedia, ? extends R> ifExternal
        ) {
            return ifLink.apply(this);
        }

        static LinkMedia parseLink(Map<String, ?> map) {
            String id = getAttrOrThrow(map, Attr.ID);
            String collection = getAttr(map, Attr.COLLECTION, String.class).orElse(null);
            LinkMedia media = new LinkMedia(id).collection(collection);
            getAttr(map, Attr.OCCURRENCE_KEY, String.class).ifPresent(media::occurrenceKey);
            return media;
        }
    }


    class ExternalMedia extends AbstractMedia<ExternalMedia> {
        private final String url;

        ExternalMedia(String url) {
            this.url = requireNonNull(url, "url");
        }

        @Override
        public ExternalMedia copy() {
            return unsafeCast(parse(toMap()));
        }

        /**
         * Returns the {@code url} value for the external media.
         */
        public String url() {
            return url;
        }

        @Override
        public MediaType type() {
            return MediaType.EXTERNAL;
        }

        @Override
        public <R> R fold(
                Function<? super FileMedia, ? extends R> ifFile,
                Function<? super LinkMedia, ? extends R> ifLink,
                Function<? super ExternalMedia, ? extends R> ifExternal
        ) {
            return ifExternal.apply(this);
        }

        @Override
        public Optional<FileMedia> file() {
            return Optional.empty();
        }

        @Override
        public Optional<LinkMedia> link() {
            return Optional.empty();
        }

        @Override
        public Optional<ExternalMedia> external() {
            return Optional.of(this);
        }

        @Override
        public Map<String, ?> toMap() {
            return mapWithType()
                    .add(Key.ATTRS, map()
                            .add(Attr.TYPE, MediaType.EXTERNAL.mediaType())
                            .add(Attr.URL, url)
                            .let(this::addCommonAttrs)
                    )
                    .let(marks::addToMap);
        }

        static ExternalMedia parseExternal(Map<String, ?> map) {
            String url = getAttrOrThrow(map, Attr.URL, String.class);
            return new ExternalMedia(url);
        }

        @Override
        protected boolean mediaEquals(ExternalMedia other) {
            return url.equals(other.url);
        }

        @Override
        protected int mediaHashCode() {
            return url.hashCode();
        }

        @Override
        protected void appendMediaFields(ToStringHelper buf) {
            buf.appendField("url", url);
        }
    }
}
