package com.atlassian.adf.model.node;

import com.atlassian.adf.model.Element;
import com.atlassian.adf.model.ex.AdfException;
import com.atlassian.adf.model.ex.mark.MarkException;
import com.atlassian.adf.model.ex.mark.MarkException.ConstraintViolation;
import com.atlassian.adf.model.mark.Mark;
import com.atlassian.adf.util.FieldMap;

import javax.annotation.Nullable;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Predicate;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import static com.atlassian.adf.model.ex.AdfException.frame;
import static com.atlassian.adf.util.Cast.unsafeCast;
import static java.util.Collections.emptyList;
import static java.util.Collections.unmodifiableCollection;
import static java.util.Collections.unmodifiableSet;

/**
 * A helper class that provides consistent management of marks for the various node types that support them.
 * Note that the methods that add/remove marks have no direct knowledge of whether the rules should be
 * updated accordingly.
 *
 * @param <M> the kinds of marks that will be stored in this mark holder
 */
@SuppressWarnings({"SameParameterValue", "UnusedReturnValue", "unused"})
class MarkHolder<M extends Mark> {
    private static final Predicate<Object> ALWAYS_FALSE = mark -> false;
    private static final Map<?, ?> EMPTY_MAP = Collections.emptyMap();

    private Map<String, M> marks = unsafeCast(EMPTY_MAP);
    private Map<String, Predicate<? super M>> rules = unsafeCast(EMPTY_MAP);

    private final int limit;

    private MarkHolder(int limit) {
        if (limit < 0) {
            throw new IllegalArgumentException("limit cannot be negative: " + limit);
        }
        this.limit = limit;
    }

    /**
     * Creates a new mark holder that does not restrict the number of marks that may be used.
     *
     * @param <M> the inferred type of marks that will be stored in this mark holder
     * @return the new mark holder
     */
    static <M extends Mark> MarkHolder<M> unlimited() {
        return new MarkHolder<>(Integer.MAX_VALUE);
    }

    /**
     * Creates a new mark holder that restricts the number of marks that may be used.
     *
     * @param limit the maximum number of marks that this mark holder will be allowed to contain
     * @param <M>   the inferred type of marks that will be stored in this mark holder
     * @return the new mark holder
     */
    static <M extends Mark> MarkHolder<M> limit(int limit) {
        return new MarkHolder<>(limit);
    }

    void add(M mark) {
        frame(AdfException.MARK_PREFIX + mark.elementType(), () -> {
            Supplier<? extends AdfException> err = addIfAllowed(mark).orElse(null);
            if (err != null) throw err.get();
            return null;
        });
    }

    Optional<Supplier<? extends ConstraintViolation>> addIfAllowed(M mark) {
        if (size() >= limit) {
            return Optional.of(() -> new MarkException.LimitReached(limit, marks));
        }

        for (Map.Entry<String, Predicate<? super M>> entry : rules.entrySet()) {
            Predicate<? super M> rule = entry.getValue();
            if (!rule.test(mark)) {
                String reason = entry.getKey();
                return Optional.of(() -> new MarkException.MarkDisallowed(reason, mark));
            }
        }

        M existing = marksRW().putIfAbsent(mark.elementType(), mark);
        if (existing != null) {
            return Optional.of(() -> new MarkException.DuplicateMarkType(mark));
        }

        return Optional.empty();
    }

    Collection<M> get() {
        return unmodifiableCollection(marks.values());
    }

    Set<String> getTypes() {
        return unmodifiableSet(marks.keySet());
    }

    boolean containsType(String type) {
        return marks.containsKey(type);
    }

    boolean containsMark(M mark) {
        return mark.equals(marks.get(mark.elementType()));
    }

    Optional<M> get(String type) {
        return Optional.ofNullable(marks.get(type));
    }

    Optional<M> remove(String type) {
        return Optional.ofNullable(marks.remove(type));
    }

    boolean removeAll(Class<? extends M> markClass) {
        return marks.values().removeIf(markClass::isInstance);
    }

    boolean remove(M mark) {
        return marks.remove(mark.elementType(), mark);
    }

    <T extends M> Stream<T> stream(Class<T> markClass) {
        return marks.values().stream()
                .filter(markClass::isInstance)
                .map(markClass::cast);
    }

    private Map<String, Predicate<? super M>> rulesRW() {
        Map<String, Predicate<? super M>> rules = this.rules;
        if (rules == EMPTY_MAP) {
            rules = new LinkedHashMap<>();
            this.rules = rules;
        }
        return rules;
    }

    private Map<String, M> marksRW() {
        var marks = this.marks;
        if (marks == EMPTY_MAP) {
            marks = new LinkedHashMap<>();
            this.marks = marks;
        }
        return marks;
    }

    /**
     * Disallows the owning node from carrying any marks at all.
     * <p>
     * For example, if a text node is placed in a code block, it is not permitted to contain any marks.
     *
     * @param reason the reason those mark classes are now disabled, used in the message for
     *               the {@code IllegalArgumentException} thrown by {@link #add(Mark)}
     * @throws IllegalStateException if marks have already been added to the node
     */
    void disable(String reason) {
        if (!marks.isEmpty()) {
            throw new MarkException.RestrictedMarkAlreadyPresent(reason, marks);
        }

        // Since this rule is going to match everything, don't bother to check any other rules.
        // If any other rules exist, remove them.
        var rules = rulesRW();
        if (rules.put(reason, ALWAYS_FALSE) == null && rules.size() > 1) {
            rules.keySet().removeIf(key -> !key.equals(reason));
        }
    }

    /**
     * Restricts this node to marks of the given type.
     *
     * @param cls    the specific mark class to be allowed
     * @param reason the reason this restriction is in force; use in the message for
     *               the {@code IllegalArgumentException} thrown by {@link #add(Mark)}
     * @throws IllegalStateException if the node has already been assigned marks that do not meet this requirement
     */
    void restrictToInstancesOf(Class<? extends Mark> cls, String reason) {
        addRule(cls::isInstance, reason);
    }

    /**
     * Restricts this node to no longer allow marks of the given type.
     *
     * @param cls    the specific mark class to be disallowed
     * @param reason the reason this restriction is in force; use in the message for
     *               the {@code IllegalArgumentException} thrown by {@link #add(Mark)}
     * @throws IllegalStateException if the node has already been assigned marks that do not meet this requirement
     */
    @SuppressWarnings("SameParameterValue")
    void rejectInstancesOf(Class<? extends Mark> cls, String reason) {
        addRule(mark -> !cls.isInstance(mark), reason);
    }

    /**
     * Restricts this node to only allow marks that match the given rule.
     *
     * @param rule   the rule to apply, returning {@code true} for marks that are allowed or {@code false} for those
     *               that should be rejected
     * @param reason the reason this restriction is in force; use in the message for
     *               the {@code IllegalArgumentException} thrown by {@link #add(Mark)}
     * @throws IllegalStateException if the node has already been assigned marks that do not meet this requirement
     */
    void addRule(Predicate<? super M> rule, String reason) {
        validateRule(reason, rule);
        rulesRW().put(reason, rule);
    }

    void validate() {
        rules.forEach(this::validateRule);
    }

    private void validateRule(String reason, Predicate<? super M> rule) {
        var marks = this.marks;
        for (M mark : marks.values()) {
            if (!rule.test(mark)) {
                throw new MarkException.MarkDisallowed(reason, mark);
            }
        }
    }

    List<Map<String, ?>> toListOfMaps() {
        if (marks.isEmpty()) return emptyList();
        return marks.values().stream()
                .map(Element::toMap)
                .collect(Collectors.toList());
    }

    void addToMap(FieldMap map) {
        if (!marks.isEmpty()) {
            map.put(Node.Key.MARKS, toListOfMaps());
        }
    }

    @SuppressWarnings("BooleanMethodIsAlwaysInverted")
    boolean isEmpty() {
        return marks.isEmpty();
    }

    int size() {
        return marks.size();
    }

    /**
     * Discards all existing marks.
     * <p>
     * Note that for nodes like {@code text} that set restrictions based on what other rules are in place,
     * this does <strong>not</strong> reset those restrictions.
     */
    void clear() {
        marks = unsafeCast(EMPTY_MAP);
    }

    @Override
    public boolean equals(@Nullable Object o) {
        return o == this || (o instanceof MarkHolder<?> && ((MarkHolder<?>) o).marks.equals(marks));
    }

    @Override
    public int hashCode() {
        return marks.hashCode();
    }

    @Override
    public String toString() {
        return "MarkHolder" + marks.values();
    }
}
