package com.atlassian.adf.model.node;

import com.atlassian.adf.model.Documentation;
import com.atlassian.adf.model.node.type.ListItemContent;
import com.atlassian.adf.model.node.type.ListNode;
import com.atlassian.adf.util.Factory;

import java.util.Map;
import java.util.stream.Stream;

import static com.atlassian.adf.model.ex.node.ListException.CannotUseNestedListAsFirstContentItem;
import static com.atlassian.adf.model.node.Paragraph.p;
import static com.atlassian.adf.util.ParserSupport.checkType;

/**
 * Represents an item in either a {@link BulletList bulletList} or an {@link OrderedList orderedList}.
 * <p>
 * Only the nodes that implement the {@link ListItemContent} marker interface are permitted as content.
 * Additionally, although {@link CodeBlock codeBlock} and {@link Paragraph paragraph} nodes are permitted
 * as content, they are not allowed to use any marks.
 * <p>
 * Finally, although {@code listItem} nodes may contain a nested {@link BulletList bulletList} or
 * {@link OrderedList orderedList}, these are disallowed for the <strong>first</strong> content item.
 * If there is nothing else to display before the nested list, use an {@link Paragraph#p() empty paragraph}.
 * <h2>Implementation Note</h2>
 * This library tries very hard to use the compiler's type system to enforce correctness wherever it can.
 * However, since {@link BulletList bulletList} and {@link OrderedList orderedList} must implement
 * {@link ListItemContent} so that they can be added as content at all, there was no way to exclude
 * them from satisfying the type system when adding the first content item. This must therefore be
 * checked at runtime and will result in an {@link CannotUseNestedListAsFirstContentItem exception}
 * when violated.
 * <h2>Example</h2>
 * <h3>Java</h3>
 * <pre>
 * {@link BulletList#ul(ListItem[]) ul}({@link #li(String) li}("Hello world"));
 * </pre>
 * <h3>ADF</h3>
 * <pre>{@code
 *   {
 *     "type": "bulletList",
 *     "content": [
 *       {
 *         "type": "listItem",
 *         "content": [
 *           {
 *             "type": "paragraph",
 *             "content": [
 *               {
 *                 "type": "text",
 *                 "text": "Hello world"
 *               }
 *             ]
 *           }
 *         ]
 *       }
 *     ]
 *   }
 * }</pre>
 * <h3>Result</h3>
 * <div style="color: rgb(23, 43, 77); background-color: #ffffff;">
 * <ul>
 *     <li><p>Hello world</p></li>
 * </ul>
 * </div>
 *
 * @see <a href="https://developer.atlassian.com/cloud/jira/platform/apis/document/nodes/listItem/">Node - listItem</a>
 */
@Documentation(
        state = Documentation.State.INCOMPLETE,
        date = "2023-07-26",
        comment = "should say 'bulletList' and 'orderedList' are disallowed for the first content item"
)
public class ListItem extends AbstractContentNode<ListItem, ListItemContent> {
    static Factory<ListItem> FACTORY = new Factory<>(Type.LIST_ITEM, ListItem.class, ListItem::parse);

    private ListItem() {
    }

    /**
     * @return a new, empty list item. At least one content item must be added to make it valid.
     */
    public static ListItem li() {
        return new ListItem();
    }

    /**
     * @return a convenient shorthand for
     * <code>{@link #li(ListItemContent) li}({@link Paragraph#p(String) p}(content))</code>
     */
    public static ListItem li(String content) {
        return li(p(content));
    }

    /**
     * @return a convenient shorthand for
     * <code>{@link #li(ListItemContent) li}({@link Paragraph#p(String[]) p}(content))</code>
     */
    public static ListItem li(String... content) {
        return li(p(content));
    }

    /**
     * @return a new list item with the given content
     */
    public static ListItem li(ListItemContent content) {
        return li().content(content);
    }

    /**
     * @return a new list item with the given content
     */
    public static ListItem li(ListItemContent... content) {
        return li().content(content);
    }

    /**
     * @return a new list item with the given content
     */
    public static ListItem li(Iterable<? extends ListItemContent> content) {
        return li().content(content);
    }

    /**
     * @return a new list item with the given content
     */
    public static ListItem li(Stream<? extends ListItemContent> content) {
        return li().content(content);
    }

    /**
     * @see #li()
     */
    public static ListItem listItem() {
        return new ListItem();
    }

    /**
     * @see #li(String)
     */
    public static ListItem listItem(String content) {
        return li(p(content));
    }

    /**
     * @see #li(String[])
     */
    public static ListItem listItem(String... content) {
        return li(p(content));
    }

    /**
     * @see #li(ListItemContent)
     */
    public static ListItem listItem(ListItemContent content) {
        return li().content(content);
    }

    /**
     * @see #li(ListItemContent[])
     */
    public static ListItem listItem(ListItemContent... content) {
        return li().content(content);
    }

    /**
     * @see #li(Iterable)
     */
    public static ListItem listItem(Iterable<? extends ListItemContent> content) {
        return li().content(content);
    }

    /**
     * @see #li(Stream)
     */
    public static ListItem listItem(Stream<? extends ListItemContent> content) {
        return li().content(content);
    }

    @Override
    public ListItem copy() {
        return parse(toMap());
    }

    @Override
    public String elementType() {
        return Type.LIST_ITEM;
    }

    @Override
    protected void contentNodeValidate() {
        validateFirstContentItemIsNotAList();
    }

    private void validateFirstContentItemIsNotAList() {
        requireNotEmpty();
        ListItemContent firstContentItem = content.get(0);
        if (firstContentItem instanceof ListNode<?>) {
            throw new CannotUseNestedListAsFirstContentItem();
        }
    }

    @Override
    public Map<String, ?> toMap() {
        validateFirstContentItemIsNotAList();
        return mapWithType()
                .let(this::addContent);
    }

    @Override
    protected void validateContentNodeForAppend(ListItemContent node) {
        super.validateContentNodeForAppend(node);

        if (node instanceof Paragraph) {
            ((Paragraph) node).disableMarks(this);
        } else if (node instanceof CodeBlock) {
            ((CodeBlock) node).disableMarks(this);
        } else if (node instanceof ListNode<?>) {
            if (content.isEmpty()) {
                throw new CannotUseNestedListAsFirstContentItem();
            }
        }
    }

    private static ListItem parse(Map<String, ?> map) {
        checkType(map, Type.LIST_ITEM);
        return li().parseRequiredContent(map, ListItemContent.class);
    }

    @Override
    public void appendPlainText(StringBuilder sb) {
        appendPlainTextContentJoinedWith('\n', sb);
    }
}
