package com.atlassian.adf.model.node;

import com.atlassian.adf.model.Documentation;
import com.atlassian.adf.model.ex.node.LayoutSectionException;
import com.atlassian.adf.model.mark.Breakout;
import com.atlassian.adf.model.mark.type.LayoutSectionMark;
import com.atlassian.adf.model.node.type.ContentNode;
import com.atlassian.adf.model.node.type.DocContent;
import com.atlassian.adf.util.Factory;

import javax.annotation.Nullable;
import java.util.Map;
import java.util.Optional;

import static com.atlassian.adf.util.ParserSupport.checkType;
import static java.util.Objects.requireNonNull;

/**
 * In Confluence, the top-level {@code doc} node is allowed to include {@code layoutSection} nodes.
 * These in turn contain exactly {@code 2} or {@code 3} {@link LayoutColumn layoutColumn} nodes
 * that wrap other content. The effect is similar to declaring a {@link Table table} table with
 * exactly one {@link TableRow tableRow} and containing that number of {@link TableCell tableCell}s within
 * that row. Layout sections cannot be nested and are only permitted at the top-level of the document.
 * <p>
 * Note: Jira does not currently support layout sections.
 *
 *
 * <h2>Example</h2>
 * <h3>Java</h3>
 * <pre><code>
 *     {@link #layoutSection(LayoutColumn, LayoutColumn) layoutSection}(
 *         {@link LayoutColumn#layoutColumn layoutColumn}(40).{@link ContentNode#content(Node[]) content}(
 *             {@link Paragraph#p(String) p}("Hello world")
 *         ),
 *         {@link LayoutColumn#layoutColumn layoutColumn}(60).{@link ContentNode#content(Node[]) content}(
 *             {@link Paragraph#p(String) p}("Hello world again")
 *         )
 *     );
 * </code></pre>
 * <h3>ADF</h3>
 * <pre>{@code
 *  {
 *    "type": "layoutSection",
 *    "content": [
 *      {
 *        "type": "layoutColumn",
 *        "attrs": {
 *          "width": 40
 *        },
 *        "content": [
 *          {
 *            "type": "paragraph",
 *            "content": [
 *              {
 *                "type": "text",
 *                "text": "Hello world"
 *              }
 *            ]
 *          }
 *        ]
 *      },
 *      {
 *        "type": "layoutColumn",
 *        "attrs": {
 *          "width": 60
 *        },
 *        "content": [
 *          {
 *            "type": "paragraph",
 *            "content": [
 *              {
 *                "type": "text",
 *                "text": "Hello world again"
 *              }
 *            ]
 *          }
 *        ]
 *      }
 *    ]
 *  }
 * }</pre>
 * <h3>Result</h3>
 * <div style="color: rgb(23, 43, 77); background-color: #ffffff;">
 * <table style="width: 100%; border: 1px;">
 *     <tr>
 *         <td style="width: 40%;">Hello world</td>
 *         <td style="width: 60%;">Hello world again</td>
 *     </tr>
 * </table>
 * </div>
 * Note: This example output is not using AtlasKit to render the layout section, so while it gives a vague
 * impression of what a "layout section" is, it does not faithfully reproduce the actual presentation in
 * Atlassian products.
 */
@Documentation(state = Documentation.State.UNDOCUMENTED, date = "2023-07-26")
public class LayoutSection
        extends AbstractMarkedContentNode<LayoutSection, LayoutColumn, LayoutSectionMark>
        implements DocContent {

    public static final int MINIMUM_LAYOUT_COLUMN_COUNT = 2;
    public static final int MAXIMUM_LAYOUT_COLUMN_COUNT = 3;

    static final Factory<LayoutSection> FACTORY = new Factory<>(
            Type.LAYOUT_SECTION,
            LayoutSection.class,
            LayoutSection::parse
    );

    private LayoutSection() {
    }

    /**
     * Creates a partially constructed layout section that will only be complete once exactly
     * {@code 2} or {@code 3} {@link LayoutColumn layout caloumns} have been added to it.
     *
     * @return the incomplete layout section node
     */
    public static Partial.NeedsFirstColumn layoutSection() {
        return new Partial.NeedsFirstColumn();
    }

    /**
     * Creates a layout section with the given two layout columns as content.
     * A third column can be added later using {@link ContentNode#content(Node)} if desired.
     *
     * @param col1 the first column
     * @param col2 the second column
     * @return the newly constructed layout section
     */
    public static LayoutSection layoutSection(LayoutColumn col1, LayoutColumn col2) {
        return new LayoutSection().content(col1, col2);
    }

    /**
     * Creates a layout section with the given three layout columns as content.
     * This is the maximum number of columns that are permitted, so no more can be added thereafter.
     *
     * @param col1 the first column
     * @param col2 the second column
     * @param col3 the third column
     * @return the newly constructed layout section
     */
    public static LayoutSection layoutSection(LayoutColumn col1, LayoutColumn col2, LayoutColumn col3) {
        return new LayoutSection().content(col1, col2, col3);
    }

    @Override
    public LayoutSection copy() {
        return parse(toMap());
    }

    @Override
    public String elementType() {
        return Type.LAYOUT_SECTION;
    }

    public Optional<Breakout> breakout() {
        return marks.stream(Breakout.class).findAny();
    }

    public LayoutSection breakout(@Nullable Breakout breakout) {
        marks.clear();
        if (breakout != null) {
            marks.add(breakout);
        }
        return this;
    }

    public LayoutSection wide() {
        return breakout(Breakout.wide());
    }

    public LayoutSection fullWidth() {
        return breakout(Breakout.fullWidth());
    }

    public LayoutSection defaultWidth() {
        return breakout(null);
    }

    @Override
    protected void validateContentNodeForAppend(LayoutColumn node) {
        if (content.size() >= MAXIMUM_LAYOUT_COLUMN_COUNT) {
            throw new LayoutSectionException.TooManyLayoutColumns(MAXIMUM_LAYOUT_COLUMN_COUNT + 1);
        }
    }

    @Override
    public Class<LayoutSectionMark> markClass() {
        return LayoutSectionMark.class;
    }

    @Override
    protected void markedContentNodeValidate() {
        requireTwoOrThreeColumns();
    }

    private void requireTwoOrThreeColumns() {
        int count = content.size();
        if (count < MINIMUM_LAYOUT_COLUMN_COUNT) {
            throw new LayoutSectionException.NotEnoughLayoutColumns(count);
        }
        if (count > MAXIMUM_LAYOUT_COLUMN_COUNT) {
            throw new LayoutSectionException.TooManyLayoutColumns(count);
        }
    }

    @Override
    public Map<String, ?> toMap() {
        requireTwoOrThreeColumns();
        return mapWithType()
                .let(this::addContent)
                .let(marks::addToMap);
    }

    private static LayoutSection parse(Map<String, ?> map) {
        checkType(map, Type.LAYOUT_SECTION);
        LayoutSection layoutSection = new LayoutSection()
                .parseOptionalContent(map, LayoutColumn.class);  // required, but we will use our own exception for it
        layoutSection.requireTwoOrThreeColumns();
        return layoutSection.parseMarks(map);
    }

    /**
     * Types that represent a partially constructed {@link LayoutSection layoutSection}.
     */
    public interface Partial {
        /**
         * A partially constructed layout section that still needs the first column to be added to it.
         */
        class NeedsFirstColumn {
            NeedsFirstColumn() {
            }

            public NeedsSecondColumn content(LayoutColumn col1) {
                return new NeedsSecondColumn(col1);
            }

            public LayoutSection content(LayoutColumn col1, LayoutColumn col2) {
                return layoutSection(col1, col2);
            }

            public LayoutSection content(LayoutColumn col1, LayoutColumn col2, LayoutColumn col3) {
                return layoutSection(col1, col2, col3);
            }
        }

        /**
         * A partially constructed layout section that still needs the second column to be added to it.
         */
        class NeedsSecondColumn {
            private final LayoutColumn col1;

            NeedsSecondColumn(LayoutColumn col1) {
                this.col1 = requireNonNull(col1, "col1");
            }

            public LayoutSection content(LayoutColumn col2) {
                return layoutSection(col1, col2);
            }

            public LayoutSection content(LayoutColumn col2, LayoutColumn col3) {
                return layoutSection(col1, col2, col3);
            }
        }
    }
}
