package com.atlassian.adf.model.node;

import com.atlassian.adf.model.Documentation;
import com.atlassian.adf.model.mark.Breakout;
import com.atlassian.adf.model.mark.type.ExpandMark;
import com.atlassian.adf.model.node.type.ContentNode;
import com.atlassian.adf.model.node.type.DocContent;
import com.atlassian.adf.model.node.type.LayoutColumnContent;
import com.atlassian.adf.model.node.type.NonNestableBlockContent;
import com.atlassian.adf.util.Factory;

import javax.annotation.Nullable;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Stream;

import static com.atlassian.adf.util.FieldMap.map;
import static com.atlassian.adf.util.ParserSupport.*;

/**
 * The {@code expand} node is a container that enables content to be hidden or shown, similar to an accordion
 * or disclosure widget.
 * <p>
 * Note: To add an expand to a table ({@code tableCell} or {@code tableHeader}) use
 * {@link NestedExpand nestedExpand} instead.
 * <h2>Example</h2>
 * <h3>Java</h3>
 * <pre><code>
 * {@link #expand(NonNestableBlockContent) expand}(
 *     {@link Paragraph#p(String) p}("Hello world")
 * );
 * </code></pre>
 * <h3>ADF</h3>
 * <pre>{@code
 *  {
 *    "type": "expand",
 *    "attrs": {
 *      "title": "Hello world"
 *    },
 *    "content": [
 *      {
 *        "type": "paragraph",
 *        "content": [
 *          {
 *            "type": "text",
 *            "text": "Hello world"
 *          }
 *        ]
 *      }
 *    ]
 *  }
 * }</pre>
 */
@Documentation(state = Documentation.State.UNDOCUMENTED, date = "2023-07-26")
public class Expand
        extends AbstractMarkedContentNode<Expand, NonNestableBlockContent, ExpandMark>
        implements DocContent, LayoutColumnContent {

    static final Factory<Expand> FACTORY = new Factory<>(Type.EXPAND, Expand.class, Expand::parse);

    private Expand() {
    }

    @Nullable
    private String title;

    public static Expand expand() {
        return new Expand();
    }

    public static Expand expand(NonNestableBlockContent content) {
        return expand().content(content);
    }

    public static Expand expand(NonNestableBlockContent... content) {
        return expand().content(content);
    }

    public static Expand expand(Iterable<? extends NonNestableBlockContent> content) {
        return expand().content(content);
    }

    public static Expand expand(Stream<? extends NonNestableBlockContent> content) {
        return expand().content(content);
    }

    public Optional<String> title() {
        return Optional.ofNullable(title);
    }

    public Expand title(String title) {
        this.title = title;
        return this;
    }

    @Override
    public Expand copy() {
        return parse(toMap());
    }

    @Override
    public String elementType() {
        return Type.EXPAND;
    }

    @Override
    public Class<ExpandMark> markClass() {
        return ExpandMark.class;
    }

    @Override
    protected void markedContentNodeValidate() {
        requireNotEmpty();
    }

    @Override
    protected void validateContentNodeForAppend(NonNestableBlockContent node) {
        super.validateContentNodeForAppend(node);
        if (node instanceof Paragraph) {
            ((Paragraph) node).disableMarks(this);
        } else if (node instanceof Heading) {
            ((Heading) node).disableMarks(this);
        } else if (node instanceof CodeBlock) {
            ((CodeBlock) node).disableMarks(this);
        }
    }

    public Optional<Breakout> breakout() {
        return marks.stream(Breakout.class).findAny();
    }

    public Expand breakout(@Nullable Breakout breakout) {
        marks.clear();
        if (breakout != null) {
            marks.add(breakout);
        }
        return this;
    }

    public Expand wide() {
        return breakout(Breakout.wide());
    }

    public Expand fullWidth() {
        return breakout(Breakout.fullWidth());
    }

    public Expand defaultWidth() {
        return breakout(null);
    }

    @Override
    protected boolean markedContentNodeEquals(Expand other) {
        return Objects.equals(title, other.title);
    }

    @Override
    protected int markedContentNodeHashCode() {
        return Objects.hashCode(title);
    }

    @Override
    protected void appendMarkedContentNodeFields(ToStringHelper buf) {
        buf.appendField("title", title);
    }

    void disableMarks(ContentNode<?, ?> parent) {
        marks.disable(parent.elementType());
    }

    // FIXME? Per the JSON schema, "expand" nodes are required to provide an "attrs" map even if it is empty.
    @Override
    public Map<String, ?> toMap() {
        requireNotEmpty();
        return mapWithType()
                .let(this::addContent)
                .add(Key.ATTRS, map().addIfPresent(Attr.TITLE, title))
                .let(marks::addToMap);
    }

    private static Expand parse(Map<String, ?> map) {
        checkType(map, Type.EXPAND);
        Expand expand = new Expand()
                .parseRequiredContent(map, NonNestableBlockContent.class);
        getOrThrow(map, Key.ATTRS);
        getAttr(map, Attr.TITLE, String.class).ifPresent(expand::title);
        return expand.parseMarks(map);
    }

}
