package com.atlassian.adf.model.node;

import com.atlassian.adf.model.Documentation;
import com.atlassian.adf.model.node.type.CaptionContent;
import com.atlassian.adf.model.node.type.InlineContent;
import com.atlassian.adf.util.Factory;

import javax.annotation.Nullable;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;

import static com.atlassian.adf.model.Element.nonEmpty;
import static com.atlassian.adf.util.FieldMap.map;
import static com.atlassian.adf.util.ParserSupport.checkType;
import static com.atlassian.adf.util.ParserSupport.getAttr;
import static com.atlassian.adf.util.ParserSupport.getAttrOrThrow;
import static java.util.Objects.requireNonNull;

/**
 * An inline node that represents an emoji.
 * <p>
 * There are three kinds of emoji:
 * <ul>
 *     <li>Standard &mdash; Unicode emoji</li>
 *     <li>Atlassian &mdash; Non-standard emoji introduced by Atlassian</li>
 *     <li>Site &mdash; Non-standard customer-defined emoji</li>
 * </ul>
 * <h2>Example: Unicode emoji</h2>
 * <h3>Java</h3>
 * <pre>
 * {@link #emoji(String) emoji}(":grinning:")
 *         .{@link #text(String) text}("&#x5C;uD83D&#x5C;uDE00");
 * </pre>
 * <h3>ADF</h3>
 * <pre>{@code
 *   {
 *     "type": "emoji",
 *     "attrs": {
 *       "shortName": ":grinning:",
 *       "text": "\uD83D\uDE00"
 *     }
 *   }
 * }</pre>
 * <h2>Example: Non-standard Atlassian emoji</h2>
 * <h3>Java</h3>
 * <pre>
 * {@link #emoji(String) emoji}(":awthanks:")
 *         .{@link #text(String) text}("&#x5C;uD83D&#x5C;uDE00");
 * </pre>
 * <h3>ADF</h3>
 * <pre>{@code
 *   {
 *     "type": "emoji",
 *     "attrs": {
 *       "shortName": ":awthanks:",
 *       "id": "atlassian-awthanks",
 *       "text": ":awthanks:"
 *     }
 *   }
 * }</pre>
 * <h2>Example: Non-standard customer emoji</h2>
 * <h3>Java</h3>
 * <pre>
 * {@link #emoji(String) emoji}(":thumbsup::skin-tone-2:")
 *         .{@link #id(String) id}("1f44d")
 *         .{@link #text(String) text}("&#x5C;uD83D&#x5C;uDC4D&#x5C;uD83C&#x5C;uDFFD");
 * </pre>
 * <h3>ADF</h3>
 * <pre>{@code
 *   {
 *     "type": "emoji",
 *     "attrs": {
 *       "shortName": ":thumbsup::skin-tone-2:",
 *       "id": "1f44d",
 *       "text": "\uD83D\uDC4D\uD83C\uDFFD"
 *     }
 *   }
 * }</pre>
 *
 * @see <a href="https://developer.atlassian.com/cloud/jira/platform/apis/document/nodes/emoji/">Node - emoji</a>
 */
@Documentation(state = Documentation.State.REVIEWED, date = "2023-07-26")
public class Emoji
        extends AbstractNode<Emoji>
        implements CaptionContent, InlineContent {

    static Factory<Emoji> FACTORY = new Factory<>(Type.EMOJI, Emoji.class, Emoji::parse);

    private String shortName;

    @Nullable
    private String id;
    @Nullable
    private String text;

    private Emoji(String shortName) {
        this.shortName = validateShortName(shortName);
    }

    public static Partial.NeedsShortName emoji() {
        return new Partial.NeedsShortName();
    }

    /**
     * @param shortName the short name that identifies this emoji, such as {@code :grinning:}
     * @return a new emoji node with the given short name
     */
    public static Emoji emoji(String shortName) {
        return new Emoji(shortName);
    }

    public Emoji shortName(String shortName) {
        this.shortName = validateShortName(shortName);
        return this;
    }

    /**
     * Returns the short name of the emoji.
     *
     * @return the short name of the emoji.
     */
    public String shortName() {
        return shortName;
    }

    /**
     * Returns the id of the emoji, if set.
     *
     * @return the id of the emoji, or {@code empty()} if not set.
     */
    public Optional<String> id() {
        return Optional.ofNullable(id);
    }

    /**
     * Sets the internal identifier for the emoji.
     *
     * @param id an identifier for the emoji with a format that varies based on the emoji type.
     *           The is is not intended to have any user-facing meaning.
     * @return {@code this}
     */
    public Emoji id(String id) {
        this.id = id;
        return this;
    }

    /**
     * Returns the text of the emoji, if set.
     *
     * @return the text of the emoji, or {@code empty()} if not set.
     */
    public Optional<String> text() {
        return Optional.ofNullable(text);
    }

    /**
     * Sets the text version of the emoji.
     *
     * @param text the text version of the emoji; if unset, the short name is used in its place
     * @return {@code this}
     */
    public Emoji text(String text) {
        this.text = text;
        return this;
    }

    @Override
    public Emoji copy() {
        return parse(toMap());
    }

    @Override
    protected boolean nodeEquals(Emoji other) {
        return shortName.equals(other.shortName)
                && Objects.equals(id, other.id)
                && Objects.equals(text, other.text);
    }

    @Override
    protected int nodeHashCode() {
        return Objects.hash(shortName, id, text);
    }

    @Override
    protected void appendNodeFields(ToStringHelper buf) {
        buf.appendTextField(text);
        buf.appendField("shortName", shortName);
        buf.appendField("id", id);
    }

    @Override
    public String elementType() {
        return Type.EMOJI;
    }

    @Override
    public void validate() {
    }

    @Override
    public Map<String, ?> toMap() {
        return mapWithType()
                .add(Key.ATTRS, map(Attr.SHORT_NAME, shortName)
                        .addIfPresent(Attr.ID, id)
                        .addIfPresent(Attr.TEXT, text)
                );
    }

    @Override
    public void appendPlainText(StringBuilder sb) {
        String text = this.text;
        sb.append((text != null) ? text : shortName);
    }

    private static Emoji parse(Map<String, ?> map) {
        checkType(map, Type.EMOJI);
        Emoji emoji = emoji(getAttrOrThrow(map, Attr.SHORT_NAME, String.class));
        getAttr(map, Attr.ID, String.class).ifPresent(emoji::id);
        getAttr(map, Attr.TEXT, String.class).ifPresent(emoji::text);
        return emoji;
    }

    private static String validateShortName(String shortName) {
        requireNonNull(shortName, "shortName");
        return nonEmpty(shortName, "shortName");
    }


    /**
     * Types that represent a partially constructed {@link Emoji emoji}.
     */
    public interface Partial {
        /**
         * This partially constructed {@code emoji} still needs its {@code shortName} attribute set.
         */
        class NeedsShortName {
            NeedsShortName() {
            }

            public Emoji shortName(String shortName) {
                return new Emoji(shortName);
            }
        }
    }

    public interface Standard {
        // These are factories rather than constants because the objects we return are mutable.

        static Emoji slightSmile() {
            return emoji(":slight_smile:").id("1f642").text("\uD83D\uDE42");
        }

        static Emoji disappointed() {
            return emoji(":disappointed:").id("1f61e").text("\uD83D\uDE1E");
        }

        static Emoji stuckOutTongue() {
            return emoji(":stuck_out_tongue:").id("1f61b").text("\uD83D\uDE1B");
        }

        static Emoji smiley() {
            return emoji(":smiley:").id("1f603").text("\uD83D\uDE03");
        }

        static Emoji wink() {
            return emoji(":wink:").id("1f609").text("\uD83D\uDE09");
        }

        static Emoji thumbsUp() {
            return emoji(":thumbsup:").id("1f44d").text("\uD83D\uDC4D");
        }

        static Emoji thumbsDown() {
            return emoji(":thumbsdown:").id("1f44e").text("\uD83D\uDC4E");
        }

        static Emoji info() {
            return emoji(":info:").id("atlassian-info").text(":info:");
        }

        static Emoji checkMark() {
            return emoji(":check_mark:").id("atlassian-check_mark").text(":check_mark:");
        }

        static Emoji crossMark() {
            return emoji(":cross_mark:").id("atlassian-cross_mark").text(":cross_mark:");
        }

        static Emoji warning() {
            return emoji(":warning:").id("atlassian-warning").text(":warning:");
        }

        static Emoji plus() {
            return emoji(":plus:").id("atlassian-plus").text(":plus:");
        }

        static Emoji minus() {
            return emoji(":minus:").id("atlassian-minus").text(":minus:");
        }

        static Emoji question() {
            return emoji(":question:").id("atlassian-question_mark").text(":question:");
        }

        static Emoji lightBulbOn() {
            return emoji(":light_bulb_on:").id("atlassian-light_bulb_on").text(":light_bulb_on:");
        }

        static Emoji lightBulbOff() {
            return emoji(":light_bulb_off:").id("atlassian-light_bulb_off").text(":light_bulb_off:");
        }

        static Emoji yellowStar() {
            return emoji(":yellow_star:").id("atlassian-yellow_star").text(":yellow_star:");
        }

        static Emoji redStar() {
            return emoji(":red_star:").id("atlassian-red_star").text(":red_star:");
        }

        static Emoji greenStar() {
            return emoji(":green_star:").id("atlassian-green_star").text(":green_star:");
        }

        static Emoji blueStar() {
            return emoji(":blue_star:").id("atlassian-blue_star").text(":blue_star:");
        }

        static Emoji flagOn() {
            return emoji(":flag_on:").id("atlassian-flag_on").text(":flag_on:");
        }

        static Emoji flagOff() {
            return emoji(":flag_off:").id("atlassian-flag_off").text(":flag_off:");
        }
    }
}
