package com.atlassian.adf.model.node;

import com.atlassian.adf.model.Documentation;
import com.atlassian.adf.model.ex.node.CardException;
import com.atlassian.adf.model.node.type.DocContent;
import com.atlassian.adf.model.node.type.LayoutColumnContent;
import com.atlassian.adf.model.node.type.NonNestableBlockContent;
import com.atlassian.adf.model.node.type.TableCellContent;
import com.atlassian.adf.util.EnumParser;
import com.atlassian.adf.util.Factory;

import javax.annotation.Nullable;
import java.net.URI;
import java.net.URL;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;

import static com.atlassian.adf.util.FieldMap.map;
import static com.atlassian.adf.util.ParserSupport.*;
import static java.util.Objects.requireNonNull;

// Note: This does not currently implement CardNode because it only supports "url", not "data".
@Documentation(state = Documentation.State.UNDOCUMENTED, date = "2023-07-26")
public class EmbedCard
        extends AbstractNode<EmbedCard>
        implements DocContent, LayoutColumnContent, NonNestableBlockContent, TableCellContent {

    static final Factory<EmbedCard> FACTORY = new Factory<>(
            Type.EMBED_CARD,
            EmbedCard.class,
            EmbedCard::parse
    );

    private Layout layout;
    private String url;

    @Nullable
    private Number originalWidth;
    @Nullable
    private Number originalHeight;
    @Nullable
    private Number width;

    private EmbedCard(Layout layout, String url) {
        this.layout = requireNonNull(layout, "layout");
        this.url = requireNonNull(url, "url");
    }

    public static EmbedCard embedCard(Layout layout, URL url) {
        requireNonNull(layout, "layout");
        requireNonNull(url, "url");
        String uri = url.toString();
        uri = cleanUri(uri, "url");
        return new EmbedCard(layout, uri);
    }

    public static EmbedCard embedCard(Layout layout, URI url) {
        requireNonNull(layout, "layout");
        requireNonNull(url, "url");
        return new EmbedCard(layout, url.toString());
    }

    public static Partial.NeedsLayout embedCard() {
        return new Partial.NeedsLayout();
    }

    public static Partial.NeedsUrl embedCard(String layout) {
        return new Partial.NeedsUrl(layout);
    }

    public static Partial.NeedsUrl embedCard(Layout layout) {
        return new Partial.NeedsUrl(layout);
    }

    public Layout layout() {
        return layout;
    }

    public EmbedCard layout(String layout) {
        return layout(Layout.PARSER.parse(layout));
    }

    public EmbedCard layout(Layout layout) {
        this.layout = requireNonNull(layout, "layout");
        return this;
    }

    public EmbedCard wide() {
        return layout(Layout.WIDE);
    }

    public EmbedCard fullWidth() {
        return layout(Layout.FULL_WIDTH);
    }

    public EmbedCard center() {
        return layout(Layout.CENTER);
    }

    public EmbedCard wrapRight() {
        return layout(Layout.WRAP_RIGHT);
    }

    public EmbedCard wrapLeft() {
        return layout(Layout.WRAP_LEFT);
    }

    public EmbedCard alignEnd() {
        return layout(Layout.ALIGN_END);
    }

    public EmbedCard alignStart() {
        return layout(Layout.ALIGN_START);
    }

    public String url() {
        return url;
    }

    public EmbedCard url(String url) {
        this.url = cleanUri(url, "url");
        return this;
    }

    public EmbedCard url(URL url) {
        this.url = cleanUri(url, "url");
        return this;
    }

    public EmbedCard url(URI url) {
        this.url = cleanUri(url, "url");
        return this;
    }

    public EmbedCard originalWidth(Number originalWidth) {
        this.originalWidth = originalWidth;
        return this;
    }

    public Optional<Number> originalWidth() {
        return Optional.ofNullable(originalWidth);
    }

    public EmbedCard originalHeight(Number originalHeight) {
        this.originalHeight = originalHeight;
        return this;
    }

    public Optional<Number> originalHeight() {
        return Optional.ofNullable(originalHeight);
    }

    public EmbedCard width(@Nullable Number width) {
        if (width != null) {
            double widthDouble = width.doubleValue();
            if (widthDouble < 0.0 || widthDouble > 100.0) {
                throw new CardException.InvalidWidth(width);
            }
            this.width = width;
        }
        return this;
    }

    public Optional<Number> width() {
        return Optional.ofNullable(width);
    }

    public EmbedCard originalSize(Number originalWidth, Number originalHeight) {
        return originalWidth(originalWidth).originalHeight(originalHeight);
    }

    @Override
    public EmbedCard copy() {
        return parse(toMap());
    }

    @Override
    public String elementType() {
        return Type.EMBED_CARD;
    }

    @Override
    protected boolean nodeEquals(EmbedCard other) {
        return layout == other.layout
                && url.equals(other.url)
                && numberEq(originalWidth, other.originalWidth)
                && numberEq(originalHeight, other.originalHeight)
                && numberEq(width, other.width);
    }

    @Override
    protected int nodeHashCode() {
        return Objects.hash(layout, url, numberHash(originalWidth), numberHash(originalHeight), numberHash(width));
    }

    @Override
    protected void appendNodeFields(ToStringHelper buf) {
        buf.appendField("layout", layout);
        buf.appendField("url", url);
        buf.appendField("originalWidth", originalWidth);
        buf.appendField("originalHeight", originalHeight);
        buf.appendField("width", width);
    }

    @Override
    public Map<String, ?> toMap() {
        return mapWithType()
                .add(Key.ATTRS, map()
                        .addMapped(Attr.LAYOUT, layout, Layout::layout)
                        .add(Attr.URL, url)
                        .addIfPresent(Attr.ORIGINAL_WIDTH, originalWidth)
                        .addIfPresent(Attr.ORIGINAL_HEIGHT, originalHeight)
                        .addIfPresent(Attr.WIDTH, width)
                );
    }

    @Override
    public void validate() {
    }

    private static EmbedCard parse(Map<String, ?> map) {
        checkType(map, Type.EMBED_CARD);
        Layout layout = Layout.PARSER.parse(getAttrOrThrow(map, Attr.LAYOUT, String.class));
        String url = getAttrOrThrow(map, Attr.URL, String.class);
        EmbedCard embedCard = new EmbedCard(layout, url);
        getAttrNumber(map, Attr.ORIGINAL_WIDTH).ifPresent(embedCard::originalWidth);
        getAttrNumber(map, Attr.ORIGINAL_HEIGHT).ifPresent(embedCard::originalHeight);
        getAttrNumber(map, Attr.WIDTH).ifPresent(embedCard::width);
        return embedCard;
    }


    public enum Layout {
        WIDE("wide"),
        FULL_WIDTH("full-width"),
        CENTER("center"),
        WRAP_RIGHT("wrap-right"),
        WRAP_LEFT("wrap-left"),
        ALIGN_END("align-end"),
        ALIGN_START("align-start");

        static final EnumParser<Layout> PARSER = new EnumParser<>(Layout.class, Layout::layout);

        private final String layout;

        Layout(String layout) {
            this.layout = layout;
        }

        public String layout() {
            return layout;
        }
    }


    /**
     * Types that represent a partially constructed {@link EmbedCard embedCard}.
     */
    public interface Partial {
        /**
         * This partially constructed {@code embedCard} still needs its {@code layout} attribute set.
         */
        class NeedsLayout {
            NeedsLayout() {
            }

            public NeedsUrl layout(String layout) {
                return new NeedsUrl(layout);
            }

            public NeedsUrl layout(Layout layout) {
                return new NeedsUrl(layout);
            }

            public NeedsUrl wide() {
                return new NeedsUrl(Layout.WIDE);
            }

            public NeedsUrl fullWidth() {
                return new NeedsUrl(Layout.FULL_WIDTH);
            }

            public NeedsUrl center() {
                return new NeedsUrl(Layout.CENTER);
            }

            public NeedsUrl wrapRight() {
                return new NeedsUrl(Layout.WRAP_RIGHT);
            }

            public NeedsUrl wrapLeft() {
                return new NeedsUrl(Layout.WRAP_LEFT);
            }

            public NeedsUrl alignEnd() {
                return new NeedsUrl(Layout.ALIGN_END);
            }

            public NeedsUrl alignStart() {
                return new NeedsUrl(Layout.ALIGN_START);
            }
        }

        /**
         * This partially constructed {@code embedCard} still needs its {@code url} attribute set.
         */
        class NeedsUrl {
            private final Layout layout;

            NeedsUrl(String layout) {
                this(Layout.PARSER.parse(layout));
            }

            NeedsUrl(Layout layout) {
                this.layout = requireNonNull(layout, "layout");
            }

            public EmbedCard url(String url) {
                return new EmbedCard(layout, cleanUri(url, "url"));
            }

            public EmbedCard url(URL url) {
                return url(requireNonNull(url, "url").toString());
            }

            public EmbedCard url(URI url) {
                return new EmbedCard(layout, requireNonNull(url, "url").toString());
            }
        }
    }
}
