package com.atlassian.adf.model.node;

import com.atlassian.adf.model.ex.node.CardException;
import com.atlassian.adf.model.node.type.CardNode;
import com.atlassian.annotations.Internal;

import javax.annotation.Nullable;
import java.net.URI;
import java.net.URL;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.function.Function;

import static com.atlassian.adf.util.Cast.unsafeCast;
import static com.atlassian.adf.util.FieldMap.map;
import static com.atlassian.adf.util.ParserSupport.cleanUri;
import static com.atlassian.adf.util.ParserSupport.getAttr;
import static java.util.Objects.requireNonNull;

@Internal
public abstract class AbstractCardNode<C extends AbstractCardNode<C>>
        extends AbstractNode<C>
        implements CardNode {

    private final UrlOrData urlOrData;

    protected AbstractCardNode(UrlOrData urlOrData) {
        this.urlOrData = requireNonNull(urlOrData, "urlOrData");
    }

    @Override
    public abstract C copy();

    @Override
    protected final boolean nodeEquals(C other) {
        return urlOrData.equals(other.urlOrData());
    }

    @Override
    protected final int nodeHashCode() {
        return urlOrData.hashCode();
    }

    @Override
    protected final void appendNodeFields(ToStringHelper buf) {
        urlOrData.accept(
                url -> buf.appendField("url", url),
                data -> buf.appendField("data", data)
        );
    }

    @Override
    public final Map<String, ?> toMap() {
        return map(
                Key.TYPE, elementType(),
                Key.ATTRS, urlOrData().fold(
                        url -> map(Attr.URL, url),
                        data -> map(Attr.DATA, map().addAll(data))
                )
        );
    }

    protected static UrlOrData parseUrlOrData(Map<String, ?> map) {
        String url = getAttr(map, Attr.URL, String.class).orElse(null);
        Map<String, ?> data = unsafeCast(getAttr(map, Attr.DATA, Map.class).orElse(null));
        if (url != null) {
            if (data != null) {
                throw new CardException.UrlAndDataCannotBothBeSet();
            }
            return new UrlImpl(url);
        }
        if (data == null) {
            throw new CardException.UrlOrDataMustBeSet();
        }
        return new DataImpl(data);
    }

    @Override
    public final UrlOrData urlOrData() {
        return urlOrData;
    }

    @Override
    public void validate() {
    }



    private static class UrlImpl implements UrlOrData {
        private final String url;

        private UrlImpl(String url) {
            this.url = requireNonNull(url, "url");
        }

        @Override
        public <T> T fold(
                Function<? super String, ? extends T> ifUrl,
                Function<? super Map<String, ?>, ? extends T> ifData
        ) {
            return ifUrl.apply(url);
        }

        @Override
        public void accept(
                Consumer<? super String> ifUrl,
                Consumer<? super Map<String, ?>> ifData
        ) {
            ifUrl.accept(url);
        }

        @Override
        public Optional<String> url() {
            return Optional.of(url);
        }

        @Override
        public Optional<Map<String, ?>> data() {
            return Optional.empty();
        }

        @Override
        public boolean equals(@Nullable Object o) {
            return this == o || (o instanceof UrlImpl && ((UrlImpl) o).url.equals(url));
        }

        @Override
        public int hashCode() {
            return getClass().hashCode() * 31 + url.hashCode();
        }

        @Override
        public String toString() {
            return "UrlImpl{url=" + url + '}';
        }
    }


    private static class DataImpl implements UrlOrData {
        private final Map<String, ?> data;

        DataImpl(Map<String, ?> data) {
            requireNonNull(data, "data");
            this.data = new LinkedHashMap<>(data);
        }

        @Override
        public <T> T fold(Function<? super String, ? extends T> ifUrl, Function<? super Map<String, ?>, ? extends T> ifData) {
            return ifData.apply(data);
        }

        @Override
        public void accept(Consumer<? super String> ifUrl, Consumer<? super Map<String, ?>> ifData) {
            ifData.accept(data);
        }

        @Override
        public Optional<String> url() {
            return Optional.empty();
        }

        @Override
        public Optional<Map<String, ?>> data() {
            return Optional.of(data);
        }

        @Override
        public boolean equals(Object o) {
            return this == o || (o instanceof DataImpl && ((DataImpl) o).data.equals(data));
        }

        @Override
        public int hashCode() {
            return getClass().hashCode() * 31 + data.hashCode();
        }

        @Override
        public String toString() {
            return "DataImpl{data=" + data + '}';
        }
    }


    /**
     * Types that represent a partially constructed {@link BlockCard blockCard} or {@link InlineCard inlineCard}.
     */
    public interface Partial {
        /**
         * This partially constructed card still needs either a {@code url} or {@code data} value set.
         *
         * @param <C> the type of card that has been partially constructed and will be fully constructed once
         *            those missing values have been supplied.
         */
        class NeedsUrlOrData<C extends AbstractCardNode<C>> {
            private final Function<UrlOrData, C> constructor;

            public NeedsUrlOrData(Function<UrlOrData, C> constructor) {
                this.constructor = constructor;
            }

            public C url(String url) {
                String cleanUrl = cleanUri(url, "url");
                return constructor.apply(new UrlImpl(cleanUrl));
            }

            public C url(URL url) {
                String cleanUrl = cleanUri(url, "url");
                return constructor.apply(new UrlImpl(cleanUrl));
            }

            public C url(URI url) {
                String cleanUrl = cleanUri(url, "url");
                return constructor.apply(new UrlImpl(cleanUrl));
            }

            public C data(Map<String, ?> data) {
                return constructor.apply(new DataImpl(data));
            }
        }
    }
}
