package com.atlassian.adf.model.mark;

import com.atlassian.adf.model.Documentation;
import com.atlassian.adf.model.ex.AdfException;
import com.atlassian.adf.model.mark.type.FormattedTextMark;
import com.atlassian.adf.model.node.Paragraph;
import com.atlassian.adf.model.node.Text;
import com.atlassian.adf.model.node.type.InlineContent;
import com.atlassian.adf.util.Factory;

import javax.annotation.concurrent.Immutable;
import java.util.Map;

import static com.atlassian.adf.model.node.Text.text;
import static com.atlassian.adf.util.ParserSupport.checkType;

/**
 * The {@code underline} mark sets <span style="text-decoration: underline;">underline</span> styling.
 * This mark applies to {@link Text} nodes and can only be combined with other
 * {@link FormattedTextMark formatted-text} marks.
 * <h2>Example</h2>
 * <h3>Java</h3>
 * <pre>
 * {@link Paragraph#p(InlineContent[]) p}(
 *         {@link Text#text(String) text}("Hello "),
 *         {@link Text#text(String) text}("world").{@link Text#underline() underline}(),
 *         {@link Text#text(String) text}("!")
 * );
 * </pre>
 * <h3>ADF</h3>
 * <pre>{@code
 *   {
 *     "type": "paragraph",
 *     "content": [
 *       {
 *         "type": "text",
 *         "text": "Hello "
 *       },
 *       {
 *         "type": "text",
 *         "text": "world",
 *         "marks": [
 *           {
 *             "type": "underline"
 *           }
 *         ]
 *       },
 *       {
 *         "type": "text",
 *         "text": "!"
 *       }
 *     ]
 *   }
 * }</pre>
 * <h3>Result</h3>
 * <div style="color: rgb(23, 43, 77); background-color: #ffffff;">
 * <p>Hello <span style="text-decoration: underline;">world</span>!</p>
 * </div>
 *
 * @see <a href="https://developer.atlassian.com/cloud/jira/platform/apis/document/marks/underline/">Mark - underline</a>
 */
@Documentation(state = Documentation.State.REVIEWED, date = "2023-07-26")
@Immutable
public class Underline
        extends AbstractMark
        implements FormattedTextMark {

    private static final Underline UNDERLINE = new Underline();
    static final Factory<Underline> FACTORY = new Factory<>(Type.UNDERLINE, Underline.class, Underline::parse);

    private Underline() {
        // singleton
    }

    /**
     * @return an {@code underline} mark
     */
    public static Underline underline() {
        return UNDERLINE;
    }

    /**
     * Convenience method for constructing a {@code text} node with an {@code underline} mark.
     *
     * @param text the text to mark
     * @return the resulting marked text node
     */
    public static Text underline(String text) {
        return text(text).underline();
    }

    /**
     * Convenience method for applying an {@code underline} mark to an existing {@code text} node.
     *
     * @param text the text to mark
     * @return the resulting marked text node
     * @throws AdfException if it isn't possible to apply that mark to the existing {@code text} node
     */
    public static Text underline(Text text) {
        return text.underline();
    }

    @Override
    public Underline copy() {
        return this;
    }

    @Override
    public String elementType() {
        return Type.UNDERLINE;
    }

    private static Underline parse(Map<String, ?> map) {
        checkType(map, Type.UNDERLINE);
        return underline();
    }
}
